<?php
/**
 * Database Setup Script for SSV Website
 * Run this file once to set up the database
 */

// Skip the database test in config/db.php during setup
define('SKIP_DB_TEST', true);

echo "<h2>SSV Database Setup</h2>";

try {
    // Connect to MySQL without selecting a database first
    $pdo = new PDO("mysql:host=localhost;charset=utf8mb4", "root", "");
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    echo "<p>✅ Connected to MySQL server</p>";
    
    // Create database if it doesn't exist
    $pdo->exec("CREATE DATABASE IF NOT EXISTS ssv_website CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");
    echo "<p>✅ Database 'ssv_website' created or already exists</p>";
    
    // Now connect to the specific database
    $pdo = new PDO("mysql:host=localhost;dbname=ssv_website;charset=utf8mb4", "root", "");
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    echo "<p>✅ Connected to ssv_website database</p>";
    
    // Read and execute the SQL schema file
    $sqlFile = __DIR__ . '/database_schema.sql';
    if (file_exists($sqlFile)) {
        $sql = file_get_contents($sqlFile);
        
        // Remove the CREATE DATABASE lines since we already did that
        $sql = preg_replace('/^CREATE DATABASE.*$/m', '', $sql);
        $sql = preg_replace('/^USE ssv_website;$/m', '', $sql);
        
        // Split into individual statements and execute
        $statements = array_filter(array_map('trim', explode(';', $sql)));
        
        $executedCount = 0;
        foreach ($statements as $statement) {
            if (!empty($statement) && !preg_match('/^--/', $statement)) {
                try {
                    $pdo->exec($statement);
                    $executedCount++;
                } catch (PDOException $e) {
                    // Ignore table already exists errors
                    if (strpos($e->getMessage(), 'already exists') === false) {
                        echo "<p>⚠️ Warning executing statement: " . htmlspecialchars($e->getMessage()) . "</p>";
                    }
                }
            }
        }
        
        echo "<p>✅ Executed {$executedCount} SQL statements</p>";
        
        // Verify tables were created
        $stmt = $pdo->query("SHOW TABLES");
        $tables = $stmt->fetchAll(PDO::FETCH_COLUMN);
        
        echo "<p>✅ Created tables: " . implode(', ', $tables) . "</p>";
        
        // Check if admin user exists
        $stmt = $pdo->query("SELECT COUNT(*) FROM admin_users");
        $adminCount = $stmt->fetchColumn();
        
        if ($adminCount > 0) {
            echo "<p>✅ Admin user already exists</p>";
        } else {
            echo "<p>⚠️ No admin users found</p>";
        }
        
        // Check sample data
        $stmt = $pdo->query("SELECT COUNT(*) FROM services WHERE is_featured = 1");
        $serviceCount = $stmt->fetchColumn();
        echo "<p>✅ Featured services: {$serviceCount}</p>";
        
        $stmt = $pdo->query("SELECT COUNT(*) FROM testimonials WHERE is_approved = 1");
        $testimonialCount = $stmt->fetchColumn();
        echo "<p>✅ Approved testimonials: {$testimonialCount}</p>";
        
        echo "<h3 style='color: green;'>✅ Database setup completed successfully!</h3>";
        echo "<p><strong>You can now access your website at:</strong> <a href='http://localhost/SSV'>http://localhost/SSV</a></p>";
        echo "<p><strong>Admin panel:</strong> <a href='http://localhost/SSV/admin'>http://localhost/SSV/admin</a></p>";
        echo "<p><strong>Default admin login:</strong></p>";
        echo "<ul><li>Username: admin</li><li>Password: admin123</li></ul>";
        
        echo "<p style='background: #f0f8ff; padding: 10px; border-left: 4px solid #0066cc;'>";
        echo "<strong>Next steps:</strong><br>";
        echo "1. Delete this setup file for security: <code>setup_database.php</code><br>";
        echo "2. Visit your website to see if it loads correctly<br>";
        echo "3. Login to admin panel to customize content<br>";
        echo "4. Add real images to replace placeholders";
        echo "</p>";
        
    } else {
        echo "<p style='color: red;'>❌ Could not find database_schema.sql file</p>";
    }
    
} catch (PDOException $e) {
    echo "<p style='color: red;'>❌ Database error: " . htmlspecialchars($e->getMessage()) . "</p>";
    echo "<p><strong>Common solutions:</strong></p>";
    echo "<ul>";
    echo "<li>Make sure XAMPP MySQL is running</li>";
    echo "<li>Check MySQL credentials in config/db.php</li>";
    echo "<li>Verify MySQL port (default 3306)</li>";
    echo "</ul>";
} catch (Exception $e) {
    echo "<p style='color: red;'>❌ Error: " . htmlspecialchars($e->getMessage()) . "</p>";
}

echo "<hr>";
echo "<p><small>After successful setup, you can delete this file for security.</small></p>";
?>