<?php
// Page configuration
$page_title = "Our Services";
$page_description = "Professional event management services for weddings, engagements, birthdays, and corporate events. Complete packages with decoration, catering, and entertainment.";
$body_class = "services-page";

// Include database connection
require_once '../config/db.php';

// Fetch all active services
$services = [];
try {
    $db = getDB();
    $stmt = $db->prepare("SELECT * FROM services WHERE is_active = 1 ORDER BY display_order, id");
    $stmt->execute();
    $services = $stmt->fetchAll();
} catch(Exception $e) {
    // Handle error silently for now
}

// Fetch service categories
$categories = [];
try {
    $stmt = $db->prepare("SELECT * FROM service_categories WHERE is_active = 1 ORDER BY name");
    $stmt->execute();
    $categories = $stmt->fetchAll();
} catch(Exception $e) {
    // Handle error silently for now
}

// Include header
include '../includes/header.php';
?>

<!-- Services Hero Section -->
<section class="section" style="background: var(--bg-gradient); color: white; padding-top: 8rem;">
    <div class="container">
        <div class="text-center animate-fade-up">
            <h1 class="section-title" style="color: white; font-size: 3rem;">Our Services</h1>
            <p class="section-subtitle script-text" style="color: var(--light-gold); font-size: 1.5rem;">Complete Event Management Solutions</p>
            <p class="section-description" style="color: rgba(255,255,255,0.9); font-size: 1.2rem; margin-bottom: 3rem;">
                From intimate gatherings to grand celebrations, we provide comprehensive event management services 
                tailored to make your special moments unforgettable.
            </p>
        </div>
    </div>
</section>

<!-- Service Categories Section -->
<section class="section" style="background: var(--bg-secondary); padding: 2rem 0;">
    <div class="container">
        <div class="category-filters text-center">
            <button class="filter-btn active" data-category="all">All Services</button>
            <button class="filter-btn" data-category="anniversary">Anniversary</button>
            <button class="filter-btn" data-category="birthday">Birthday</button>
            <button class="filter-btn" data-category="corporate">Corporate</button>
            <button class="filter-btn" data-category="engagement">Engagement</button>
            <button class="filter-btn" data-category="reception">Reception</button>
            <button class="filter-btn" data-category="wedding">Wedding</button>
        </div>
    </div>
</section>

<!-- Services Grid Section -->
<section class="section">
    <div class="container">
        <div class="services-grid">
            <?php if (empty($services)): ?>
                <!-- Default services if no data from database -->
                <div class="service-item card glass-card animate-fade-up" data-category="wedding">
                    <div class="service-image">
                        <img src="assets/images/services/wedding-service.jpg" alt="Wedding Planning Service" 
                             data-src="assets/images/services/wedding-service.jpg" class="lazy">
                        <div class="service-overlay">
                            <div class="service-icon">
                                <i class="fas fa-heart"></i>
                            </div>
                        </div>
                    </div>
                    <div class="service-content">
                        <h3 class="service-title">Royal Wedding Package</h3>
                        <p class="service-description">
                            Complete wedding arrangement with premium decoration, multi-cuisine catering, professional photography, 
                            videography, and entertainment. Includes mandap decoration, flower arrangements, lighting, sound system, 
                            and dedicated wedding coordinator.
                        </p>
                        <div class="service-features">
                            <div class="feature-item">
                                <i class="fas fa-users"></i>
                                <span>Up to 500 guests</span>
                            </div>
                            <div class="feature-item">
                                <i class="fas fa-utensils"></i>
                                <span>Multi-cuisine catering</span>
                            </div>
                            <div class="feature-item">
                                <i class="fas fa-camera"></i>
                                <span>Photography & Videography</span>
                            </div>
                            <div class="feature-item">
                                <i class="fas fa-music"></i>
                                <span>DJ & Entertainment</span>
                            </div>
                        </div>
                        <div class="service-price">
                            <span class="price-label">Starting from</span>
                            <div class="price-value">₹899/plate</div>
                        </div>
                        <div class="service-actions">
                            <a href="contact.php?service=wedding" class="btn btn-primary">Get Quote</a>
                            <a href="#wedding-details" class="btn btn-outline">View Details</a>
                        </div>
                    </div>
                </div>

                <div class="service-item card glass-card animate-fade-up" data-category="engagement">
                    <div class="service-image">
                        <img src="assets/images/services/engagement-service.jpg" alt="Engagement Ceremony Service" 
                             data-src="assets/images/services/engagement-service.jpg" class="lazy">
                        <div class="service-overlay">
                            <div class="service-icon">
                                <i class="fas fa-ring"></i>
                            </div>
                        </div>
                    </div>
                    <div class="service-content">
                        <h3 class="service-title">Elegant Engagement Ceremony</h3>
                        <p class="service-description">
                            Beautiful engagement setup with exquisite floral arrangements, stage decoration, lighting, 
                            catering services, and photography. Perfect for intimate engagement celebrations with family and friends.
                        </p>
                        <div class="service-features">
                            <div class="feature-item">
                                <i class="fas fa-users"></i>
                                <span>Up to 200 guests</span>
                            </div>
                            <div class="feature-item">
                                <i class="fas fa-seedling"></i>
                                <span>Floral arrangements</span>
                            </div>
                            <div class="feature-item">
                                <i class="fas fa-lightbulb"></i>
                                <span>Premium lighting</span>
                            </div>
                            <div class="feature-item">
                                <i class="fas fa-camera"></i>
                                <span>Professional photography</span>
                            </div>
                        </div>
                        <div class="service-price">
                            <span class="price-label">Starting from</span>
                            <div class="price-value">₹699/plate</div>
                        </div>
                        <div class="service-actions">
                            <a href="contact.php?service=engagement" class="btn btn-primary">Get Quote</a>
                            <a href="#engagement-details" class="btn btn-outline">View Details</a>
                        </div>
                    </div>
                </div>

                <div class="service-item card glass-card animate-fade-up" data-category="reception">
                    <div class="service-image">
                        <img src="assets/images/services/reception-service.jpg" alt="Reception Party Service" 
                             data-src="assets/images/services/reception-service.jpg" class="lazy">
                        <div class="service-overlay">
                            <div class="service-icon">
                                <i class="fas fa-glass-cheers"></i>
                            </div>
                        </div>
                    </div>
                    <div class="service-content">
                        <h3 class="service-title">Grand Reception Party</h3>
                        <p class="service-description">
                            Luxurious reception arrangement with premium decoration, multi-cuisine buffet, live music, 
                            dance floor, cocktail service, and entertainment. Create unforgettable memories for your guests.
                        </p>
                        <div class="service-features">
                            <div class="feature-item">
                                <i class="fas fa-users"></i>
                                <span>Up to 800 guests</span>
                            </div>
                            <div class="feature-item">
                                <i class="fas fa-cocktail"></i>
                                <span>Cocktail service</span>
                            </div>
                            <div class="feature-item">
                                <i class="fas fa-music"></i>
                                <span>Live entertainment</span>
                            </div>
                            <div class="feature-item">
                                <i class="fas fa-utensils"></i>
                                <span>Premium buffet</span>
                            </div>
                        </div>
                        <div class="service-price">
                            <span class="price-label">Starting from</span>
                            <div class="price-value">₹999/plate</div>
                        </div>
                        <div class="service-actions">
                            <a href="contact.php?service=reception" class="btn btn-primary">Get Quote</a>
                            <a href="#reception-details" class="btn btn-outline">View Details</a>
                        </div>
                    </div>
                </div>

                <div class="service-item card glass-card animate-fade-up" data-category="birthday">
                    <div class="service-image">
                        <img src="assets/images/services/birthday-service.jpg" alt="Birthday Party Service" 
                             data-src="assets/images/services/birthday-service.jpg" class="lazy">
                        <div class="service-overlay">
                            <div class="service-icon">
                                <i class="fas fa-birthday-cake"></i>
                            </div>
                        </div>
                    </div>
                    <div class="service-content">
                        <h3 class="service-title">Birthday Celebration</h3>
                        <p class="service-description">
                            Fun-filled birthday party arrangements with themed decoration, entertainment, games, 
                            delicious cake and food. Perfect for all age groups from kids to adults with customized themes.
                        </p>
                        <div class="service-features">
                            <div class="feature-item">
                                <i class="fas fa-users"></i>
                                <span>Up to 150 guests</span>
                            </div>
                            <div class="feature-item">
                                <i class="fas fa-palette"></i>
                                <span>Themed decoration</span>
                            </div>
                            <div class="feature-item">
                                <i class="fas fa-gamepad"></i>
                                <span>Games & activities</span>
                            </div>
                            <div class="feature-item">
                                <i class="fas fa-birthday-cake"></i>
                                <span>Custom cake</span>
                            </div>
                        </div>
                        <div class="service-price">
                            <span class="price-label">Starting from</span>
                            <div class="price-value">₹499/plate</div>
                        </div>
                        <div class="service-actions">
                            <a href="contact.php?service=birthday" class="btn btn-primary">Get Quote</a>
                            <a href="#birthday-details" class="btn btn-outline">View Details</a>
                        </div>
                    </div>
                </div>

                <div class="service-item card glass-card animate-fade-up" data-category="corporate">
                    <div class="service-image">
                        <img src="assets/images/services/corporate-service.jpg" alt="Corporate Event Service" 
                             data-src="assets/images/services/corporate-service.jpg" class="lazy">
                        <div class="service-overlay">
                            <div class="service-icon">
                                <i class="fas fa-briefcase"></i>
                            </div>
                        </div>
                    </div>
                    <div class="service-content">
                        <h3 class="service-title">Corporate Event Management</h3>
                        <p class="service-description">
                            Professional corporate event management for conferences, seminars, product launches, 
                            team building events, and annual meetings with modern facilities and professional coordination.
                        </p>
                        <div class="service-features">
                            <div class="feature-item">
                                <i class="fas fa-users"></i>
                                <span>Up to 300 guests</span>
                            </div>
                            <div class="feature-item">
                                <i class="fas fa-desktop"></i>
                                <span>AV equipment</span>
                            </div>
                            <div class="feature-item">
                                <i class="fas fa-microphone"></i>
                                <span>Sound system</span>
                            </div>
                            <div class="feature-item">
                                <i class="fas fa-coffee"></i>
                                <span>Refreshments</span>
                            </div>
                        </div>
                        <div class="service-price">
                            <span class="price-label">Starting from</span>
                            <div class="price-value">₹799/person</div>
                        </div>
                        <div class="service-actions">
                            <a href="contact.php?service=corporate" class="btn btn-primary">Get Quote</a>
                            <a href="#corporate-details" class="btn btn-outline">View Details</a>
                        </div>
                    </div>
                </div>

                <div class="service-item card glass-card animate-fade-up" data-category="anniversary">
                    <div class="service-image">
                        <img src="assets/images/services/anniversary-service.jpg" alt="Anniversary Celebration Service" 
                             data-src="assets/images/services/anniversary-service.jpg" class="lazy">
                        <div class="service-overlay">
                            <div class="service-icon">
                                <i class="fas fa-heart"></i>
                            </div>
                        </div>
                    </div>
                    <div class="service-content">
                        <h3 class="service-title">Anniversary Celebration</h3>
                        <p class="service-description">
                            Romantic anniversary celebration setup with elegant decoration, intimate dining, 
                            soft lighting, and personalized touches to make your milestone celebration memorable and special.
                        </p>
                        <div class="service-features">
                            <div class="feature-item">
                                <i class="fas fa-users"></i>
                                <span>Intimate gatherings</span>
                            </div>
                            <div class="feature-item">
                                <i class="fas fa-heart"></i>
                                <span>Romantic setup</span>
                            </div>
                            <div class="feature-item">
                                <i class="fas fa-utensils"></i>
                                <span>Fine dining</span>
                            </div>
                            <div class="feature-item">
                                <i class="fas fa-star"></i>
                                <span>Personal touches</span>
                            </div>
                        </div>
                        <div class="service-price">
                            <span class="price-label">Starting from</span>
                            <div class="price-value">₹599/person</div>
                        </div>
                        <div class="service-actions">
                            <a href="contact.php?service=anniversary" class="btn btn-primary">Get Quote</a>
                            <a href="#anniversary-details" class="btn btn-outline">View Details</a>
                        </div>
                    </div>
                </div>

            <?php else: ?>
                <!-- Dynamic services from database -->
                <?php foreach ($services as $service): ?>
                <?php 
                    // Determine category based on service title and description
                    $category = 'all';
                    $searchText = strtolower($service['title'] . ' ' . $service['description']);
                    
                    // More comprehensive keyword matching
                    if (preg_match('/\b(wedding|marriage|matrimony|shaadi|vivah)\b/i', $searchText)) {
                        $category = 'wedding';
                    } elseif (preg_match('/\b(engagement|ring ceremony|sagai|mangni)\b/i', $searchText)) {
                        $category = 'engagement';
                    } elseif (preg_match('/\b(reception|party|celebration|function)\b/i', $searchText)) {
                        $category = 'reception';
                    } elseif (preg_match('/\b(birthday|birth day|janamdin)\b/i', $searchText)) {
                        $category = 'birthday';
                    } elseif (preg_match('/\b(corporate|business|conference|seminar|meeting|office)\b/i', $searchText)) {
                        $category = 'corporate';
                    } elseif (preg_match('/\b(anniversary|vardhapti)\b/i', $searchText)) {
                        $category = 'anniversary';
                    }
                ?>
                <div class="service-item card glass-card animate-fade-up" data-category="<?php echo $category; ?>">
                    <?php if ($service['image']): ?>
                    <div class="service-image">
                        <img src="../uploads/services/<?php echo htmlspecialchars($service['image']); ?>" 
                             alt="<?php echo htmlspecialchars($service['title']); ?>"
                             data-src="../uploads/services/<?php echo htmlspecialchars($service['image']); ?>" 
                             class="lazy">
                        <div class="service-overlay">
                            <div class="service-icon">
                                <i class="fas fa-star"></i>
                            </div>
                        </div>
                    </div>
                    <?php endif; ?>
                    <div class="service-content">
                        <h3 class="service-title"><?php echo htmlspecialchars($service['title']); ?></h3>
                        <p class="service-description">
                            <?php echo htmlspecialchars($service['description']); ?>
                        </p>
                        <?php if ($service['capacity'] || $service['per_plate_charge']): ?>
                        <div class="service-features">
                            <?php if ($service['capacity']): ?>
                            <div class="feature-item">
                                <i class="fas fa-users"></i>
                                <span>Up to <?php echo $service['capacity']; ?> guests</span>
                            </div>
                            <?php endif; ?>
                        </div>
                        <?php endif; ?>
                        <?php if ($service['per_plate_charge']): ?>
                        <div class="service-price">
                            <span class="price-label">Starting from</span>
                            <div class="price-value">₹<?php echo number_format($service['per_plate_charge'], 0); ?>/plate</div>
                        </div>
                        <?php endif; ?>
                        <div class="service-actions">
                            <a href="contact.php?service=<?php echo urlencode($service['title']); ?>" class="btn btn-primary">Get Quote</a>
                        </div>
                    </div>
                </div>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>
    </div>
</section>

<!-- Why Choose Us Section -->
<section class="section" style="background: linear-gradient(135deg, #F9FAFB 0%, #F3F4F6 100%);">
    <div class="container">
        <div class="section-header animate-fade-up">
            <p class="section-subtitle script-text">Why Choose Us</p>
            <h2 class="section-title">What Makes Us Special</h2>
        </div>
        
        <div class="grid grid-3">
            <div class="card glass-card animate-fade-up" style="text-align: center; padding: 2rem;">
                <div style="font-size: 3rem; color: var(--primary-gold); margin-bottom: 1rem;">
                    <i class="fas fa-medal"></i>
                </div>
                <h4 style="margin-bottom: 1rem; color: var(--text-primary);">8+ Years Experience</h4>
                <p style="color: var(--text-secondary); margin: 0; line-height: 1.6;">
                    Over 8 years of experience in creating memorable events with attention to every detail 
                    and professional service excellence.
                </p>
            </div>
            
            <div class="card glass-card animate-fade-up" style="text-align: center; padding: 2rem;">
                <div style="font-size: 3rem; color: var(--primary-gold); margin-bottom: 1rem;">
                    <i class="fas fa-users"></i>
                </div>
                <h4 style="margin-bottom: 1rem; color: var(--text-primary);">500+ Happy Clients</h4>
                <p style="color: var(--text-secondary); margin: 0; line-height: 1.6;">
                    Successfully organized over 500 events with 100% customer satisfaction 
                    and countless happy memories created.
                </p>
            </div>
            
            <div class="card glass-card animate-fade-up" style="text-align: center; padding: 2rem;">
                <div style="font-size: 3rem; color: var(--primary-gold); margin-bottom: 1rem;">
                    <i class="fas fa-handshake"></i>
                </div>
                <h4 style="margin-bottom: 1rem; color: var(--text-primary);">Professional Team</h4>
                <p style="color: var(--text-secondary); margin: 0; line-height: 1.6;">
                    Dedicated team of creative professionals, coordinators, and specialists 
                    working together to make your event perfect.
                </p>
            </div>
        </div>
    </div>
</section>

<!-- Call to Action Section -->
<section class="section" style="background: var(--bg-gradient); color: white;">
    <div class="container">
        <div class="text-center animate-fade-up">
            <h2 class="section-title" style="color: white;">Ready to Plan Your Event?</h2>
            <p style="font-size: 1.2rem; margin-bottom: 2rem; color: rgba(255,255,255,0.9);">
                Let's discuss your requirements and create a customized package for your special occasion.
            </p>
            <div class="cta-buttons" style="display: flex; justify-content: center; gap: 1rem; flex-wrap: wrap;">
                <a href="contact.php" class="btn btn-secondary btn-lg">
                    <i class="fas fa-envelope"></i>
                    Get Free Quote
                </a>
                <a href="tel:+919876543210" class="btn btn-outline btn-lg" style="border-color: white; color: white;">
                    <i class="fas fa-phone"></i>
                    Call Us Now
                </a>
            </div>
        </div>
    </div>
</section>

<style>
/* Services page specific styles */
.text-center { text-align: center; }

.category-filters {
    margin-bottom: 2rem;
}

.filter-btn {
    padding: 0.5rem 1rem;
    margin: 0.25rem;
    background: var(--glass-white);
    border: 1px solid var(--glass-border);
    border-radius: var(--radius-full);
    font-family: var(--font-body);
    font-weight: 500;
    color: var(--text-primary);
    cursor: pointer;
    transition: var(--transition-normal);
}

.filter-btn:hover,
.filter-btn.active {
    background: var(--primary-purple);
    color: white;
    border-color: var(--primary-purple);
}

.services-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(350px, 1fr));
    gap: 2rem;
}

.service-item {
    overflow: hidden;
    transition: var(--transition-normal);
}

.service-image {
    position: relative;
    aspect-ratio: 16/9;
    overflow: hidden;
}

.service-image img {
    width: 100%;
    height: 100%;
    object-fit: cover;
    transition: var(--transition-slow);
}

.service-item:hover .service-image img {
    transform: scale(1.1);
}

.service-overlay {
    position: absolute;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(0, 0, 0, 0.6);
    display: flex;
    align-items: center;
    justify-content: center;
    opacity: 0;
    transition: var(--transition-normal);
}

.service-item:hover .service-overlay {
    opacity: 1;
}

.service-overlay .service-icon {
    width: 60px;
    height: 60px;
    background: var(--primary-gold);
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-size: 1.5rem;
}

.service-content {
    padding: 2rem;
}

.service-title {
    font-size: 1.5rem;
    margin-bottom: 1rem;
    color: var(--text-primary);
}

.service-description {
    margin-bottom: 1.5rem;
    color: var(--text-secondary);
    line-height: 1.6;
}

.service-features {
    margin-bottom: 1.5rem;
}

.feature-item {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    margin-bottom: 0.5rem;
    font-size: 0.9rem;
    color: var(--text-secondary);
}

.feature-item i {
    color: var(--primary-gold);
    font-size: 0.8rem;
    width: 16px;
}

.service-price {
    background: var(--glass-white);
    padding: 1rem;
    border-radius: var(--radius-md);
    margin-bottom: 1.5rem;
    border: 1px solid var(--glass-border);
    text-align: center;
}

.price-label {
    font-size: 0.9rem;
    color: var(--text-muted);
    display: block;
    margin-bottom: 0.5rem;
}

.price-value {
    font-size: 1.5rem;
    font-weight: 700;
    color: var(--primary-purple);
}

.service-actions {
    display: flex;
    gap: 1rem;
    justify-content: center;
}

.cta-buttons {
    display: flex;
    justify-content: center;
    gap: 1rem;
    flex-wrap: wrap;
}

/* Hidden service items during filtering */
.service-item.hidden {
    opacity: 0;
    transform: scale(0.9);
    transition: opacity 0.3s ease, transform 0.3s ease;
}

.service-item:not(.hidden) {
    opacity: 1;
    transform: scale(1);
    transition: opacity 0.3s ease, transform 0.3s ease;
}

@media (max-width: 768px) {
    .services-grid {
        grid-template-columns: 1fr;
        gap: 1.5rem;
    }
    
    .service-actions {
        flex-direction: column;
    }
    
    .service-actions .btn {
        width: 100%;
    }
    
    .cta-buttons {
        flex-direction: column;
        align-items: center;
    }
    
    .cta-buttons .btn {
        width: 100%;
        max-width: 300px;
    }
}
</style>

<script>
// Service filtering functionality
document.addEventListener('DOMContentLoaded', function() {
    const filterButtons = document.querySelectorAll('.category-filters .filter-btn');
    const serviceItems = document.querySelectorAll('.service-item');
    
    if (filterButtons.length === 0 || serviceItems.length === 0) {
        return;
    }
    
    // Initialize all items as visible
    serviceItems.forEach(item => {
        item.style.display = 'block';
        item.classList.remove('hidden');
    });
    
    filterButtons.forEach(button => {
        button.addEventListener('click', function(e) {
            e.preventDefault();
            const category = this.getAttribute('data-category');
            
            // Remove active class from all buttons
            filterButtons.forEach(btn => btn.classList.remove('active'));
            
            // Add active class to clicked button
            this.classList.add('active');
            
            // Scroll the active button into view on mobile/tablet
            if (window.innerWidth <= 768) {
                setTimeout(() => {
                    const container = this.parentElement;
                    const containerRect = container.getBoundingClientRect();
                    const buttonRect = this.getBoundingClientRect();
                    const scrollLeft = this.offsetLeft - (containerRect.width / 2) + (buttonRect.width / 2);
                    
                    container.scrollTo({
                        left: scrollLeft,
                        behavior: 'smooth'
                    });
                }, 100);
            }
            
            // Filter service items with animation
            serviceItems.forEach(item => {
                const itemCategory = item.getAttribute('data-category');
                
                if (category === 'all' || itemCategory === category) {
                    // Show item
                    item.classList.remove('hidden');
                    item.style.display = 'block';
                    // Force reflow for smooth animation
                    item.offsetHeight;
                } else {
                    // Hide item
                    item.classList.add('hidden');
                    setTimeout(() => {
                        if (item.classList.contains('hidden')) {
                            item.style.display = 'none';
                        }
                    }, 300);
                }
            });
        });
    });
});
</script>

<?php include '../includes/footer.php'; ?>
