-- SSV Website Database Schema
-- Shree Shyam Vartika Event Management

-- Create database
CREATE DATABASE IF NOT EXISTS ssv_website CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
USE ssv_website;

-- Admin users table
CREATE TABLE admin_users (
    id INT PRIMARY KEY AUTO_INCREMENT,
    username VARCHAR(50) UNIQUE NOT NULL,
    password VARCHAR(255) NOT NULL,
    email VARCHAR(100) UNIQUE NOT NULL,
    full_name VARCHAR(100) NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    last_login TIMESTAMP NULL
);

-- Services table
CREATE TABLE services (
    id INT PRIMARY KEY AUTO_INCREMENT,
    title VARCHAR(100) NOT NULL,
    description TEXT NOT NULL,
    short_description VARCHAR(255),
    image VARCHAR(255),
    capacity INT DEFAULT NULL,
    per_plate_charge DECIMAL(10,2) DEFAULT NULL,
    is_featured BOOLEAN DEFAULT FALSE,
    is_active BOOLEAN DEFAULT TRUE,
    display_order INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

-- Gallery table
CREATE TABLE gallery (
    id INT PRIMARY KEY AUTO_INCREMENT,
    title VARCHAR(100),
    description TEXT,
    file_name VARCHAR(255) NOT NULL,
    file_type ENUM('image', 'video') NOT NULL,
    file_size INT,
    service_category VARCHAR(50),
    alt_text VARCHAR(255),
    is_featured BOOLEAN DEFAULT FALSE,
    display_order INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

-- Testimonials table
CREATE TABLE testimonials (
    id INT PRIMARY KEY AUTO_INCREMENT,
    client_name VARCHAR(100) NOT NULL,
    client_email VARCHAR(100),
    rating INT DEFAULT 5 CHECK (rating >= 1 AND rating <= 5),
    feedback TEXT NOT NULL,
    event_type VARCHAR(50),
    event_date DATE,
    is_approved BOOLEAN DEFAULT FALSE,
    show_on_home BOOLEAN DEFAULT FALSE,
    display_order INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

-- Enquiries/Contact form submissions
CREATE TABLE enquiries (
    id INT PRIMARY KEY AUTO_INCREMENT,
    name VARCHAR(100) NOT NULL,
    email VARCHAR(100) NOT NULL,
    phone VARCHAR(20) NOT NULL,
    service_interested VARCHAR(100),
    event_date DATE,
    guest_count INT,
    message TEXT NOT NULL,
    status ENUM('new', 'read', 'responded', 'archived') DEFAULT 'new',
    notes TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

-- Business settings table
CREATE TABLE business_settings (
    id INT PRIMARY KEY AUTO_INCREMENT,
    setting_key VARCHAR(50) UNIQUE NOT NULL,
    setting_value TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

-- Service categories table
CREATE TABLE service_categories (
    id INT PRIMARY KEY AUTO_INCREMENT,
    name VARCHAR(50) UNIQUE NOT NULL,
    description VARCHAR(255),
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Hero slides table for home page slider
CREATE TABLE hero_slides (
    id INT PRIMARY KEY AUTO_INCREMENT,
    title VARCHAR(100) NOT NULL,
    subtitle VARCHAR(150),
    description TEXT,
    image_file VARCHAR(255) NOT NULL,
    button_text VARCHAR(50) DEFAULT 'Plan Your Event',
    button_link VARCHAR(255) DEFAULT 'contact.php',
    is_active BOOLEAN DEFAULT TRUE,
    display_order INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

-- Insert default admin user (password: admin123)
INSERT INTO admin_users (username, password, email, full_name) VALUES 
('admin', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'admin@ssv.com', 'SSV Administrator');

-- Insert default business settings
INSERT INTO business_settings (setting_key, setting_value) VALUES 
('business_name', 'Shree Shyam Vartika'),
('business_tagline', 'Making Your Special Day Unforgettable'),
('business_phone', '+91-9876543210'),
('business_email', 'info@shreeshyamvartika.com'),
('business_address', '123 Event Street, Wedding City, State 123456'),
('whatsapp_number', '+91-9876543210'),
('google_maps_embed', 'https://www.google.com/maps/embed?pb=!1m18!1m12!1m3!1d3024.123456789!2d-74.00597238459394!3d40.712775279337!2m3!1f0!2f0!3f0!3m2!1i1024!2i768!4f13.1!3m3!1m2!1s0x0%3A0x0!2zNDDCsDQyJzQ2LjAiTiA3NMKwMDAnMjEuNSJX!5e0!3m2!1sen!2sin!4v1234567890123!5m2!1sen!2sin'),
('facebook_url', 'https://facebook.com/shreeshyamvartika'),
('instagram_url', 'https://instagram.com/shreeshyamvartika'),
('youtube_url', 'https://youtube.com/@shreeshyamvartika');

-- Insert default service categories
INSERT INTO service_categories (name, description) VALUES 
('Wedding', 'Wedding ceremonies and receptions'),
('Engagement', 'Engagement ceremonies and parties'),
('Birthday', 'Birthday parties and celebrations'),
('Corporate', 'Corporate events and meetings'),
('Reception', 'Reception parties and gatherings'),
('Anniversary', 'Anniversary celebrations');

-- Insert sample services
INSERT INTO services (title, description, short_description, capacity, per_plate_charge, is_featured, display_order) VALUES 
('Royal Wedding Package', 'Complete wedding arrangement with decoration, catering, photography, and entertainment. Includes mandap decoration, flower arrangements, lighting, sound system, and professional coordination.', 'Complete wedding solution with premium decoration and catering', 500, 899.00, TRUE, 1),
('Elegant Engagement Ceremony', 'Beautiful engagement setup with floral arrangements, stage decoration, catering services, and music system. Perfect for intimate engagement celebrations.', 'Elegant engagement setup with premium decoration', 200, 699.00, TRUE, 2),
('Grand Reception Party', 'Luxurious reception arrangement with premium decoration, multi-cuisine catering, live music, and dance floor. Create unforgettable memories for your guests.', 'Grand reception with luxury decoration and entertainment', 800, 999.00, TRUE, 3),
('Birthday Bash Celebration', 'Fun-filled birthday party arrangements with themed decoration, entertainment, games, and delicious food. Perfect for all age groups from kids to adults.', 'Themed birthday parties with entertainment and catering', 150, 499.00, FALSE, 4),
('Corporate Event Management', 'Professional corporate event management including conferences, seminars, product launches, and team building events with modern facilities.', 'Professional corporate events with modern facilities', 300, 799.00, FALSE, 5);

-- Insert sample testimonials
INSERT INTO testimonials (client_name, rating, feedback, event_type, show_on_home, is_approved, display_order) VALUES 
('Rahul & Priya Sharma', 5, 'SSV made our wedding day absolutely perfect! The decoration was stunning and the coordination was flawless. Highly recommended for anyone planning their special day.', 'Wedding', TRUE, TRUE, 1),
('Amit Patel', 5, 'Organized my daughters engagement ceremony beautifully. The team was professional and attentive to every detail. Everyone praised the arrangements!', 'Engagement', TRUE, TRUE, 2),
('Sunita Enterprises', 4, 'Excellent corporate event management. They handled our annual conference with 300+ attendees seamlessly. Great food and perfect arrangements.', 'Corporate', FALSE, TRUE, 3),
('Ravi Kumar', 5, 'Best birthday party ever! My son was so happy with the superhero theme decoration. The entertainment kept all kids engaged throughout the event.', 'Birthday', TRUE, TRUE, 4);

-- Insert sample hero slides
INSERT INTO hero_slides (title, subtitle, description, image_file, button_text, button_link, display_order) VALUES 
('Shree Shyam Vartika', 'Making Your Special Day Unforgettable', 'From intimate engagements to grand weddings, from corporate events to birthday celebrations, we create magical moments with elegance, style, and perfection.', 'hero-wedding.jpg', 'Plan Your Event', 'contact.php', 1),
('Royal Wedding Celebrations', 'Where Dreams Come True', 'Experience the magic of your perfect wedding with our comprehensive wedding packages. Every detail crafted to perfection for your special day.', 'hero-wedding-2.jpg', 'View Wedding Packages', 'services.php', 2),
('Elegant Event Management', 'Creating Unforgettable Memories', 'From birthdays to corporate events, we bring creativity and professionalism to every celebration. Let us make your event extraordinary.', 'hero-events.jpg', 'Explore Services', 'services.php', 3);

-- Create indexes for better performance
CREATE INDEX idx_services_featured ON services(is_featured, display_order);
CREATE INDEX idx_gallery_category ON gallery(service_category);
CREATE INDEX idx_testimonials_home ON testimonials(show_on_home, display_order);
CREATE INDEX idx_enquiries_status ON enquiries(status, created_at);
CREATE INDEX idx_gallery_type ON gallery(file_type);
CREATE INDEX idx_hero_slides_active ON hero_slides(is_active, display_order);
