<?php
session_start();

// Include required files
require_once '../config/db.php';
require_once 'auth.php';

// Check if admin is logged in
requireAdminLogin();

// Get admin info
$admin = getCurrentAdmin();

$message = '';
$message_type = '';

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $db = getDB();
    
    // Handle logout
    if (isset($_POST['logout'])) {
        adminLogout();
    }
    
    // Handle add/edit service
    if (isset($_POST['save_service'])) {
        $service_id = $_POST['service_id'] ?? 0;
        $title = sanitizeInput($_POST['title'] ?? '');
        $description = sanitizeInput($_POST['description'] ?? '');
        $short_description = sanitizeInput($_POST['short_description'] ?? '');
        $capacity = intval($_POST['capacity'] ?? 0) ?: null;
        $per_plate_charge = floatval($_POST['per_plate_charge'] ?? 0) ?: null;
        $is_featured = isset($_POST['is_featured']) ? 1 : 0;
        $is_active = isset($_POST['is_active']) ? 1 : 0;
        $display_order = intval($_POST['display_order'] ?? 0);
        
        try {
            if ($service_id > 0) {
                $stmt = $db->prepare("UPDATE services SET title = ?, description = ?, short_description = ?, capacity = ?, per_plate_charge = ?, is_featured = ?, is_active = ?, display_order = ? WHERE id = ?");
                $stmt->execute([$title, $description, $short_description, $capacity, $per_plate_charge, $is_featured, $is_active, $display_order, $service_id]);
                $message = 'Service updated successfully!';
            } else {
                $stmt = $db->prepare("INSERT INTO services (title, description, short_description, capacity, per_plate_charge, is_featured, is_active, display_order) VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
                $stmt->execute([$title, $description, $short_description, $capacity, $per_plate_charge, $is_featured, $is_active, $display_order]);
                $message = 'Service added successfully!';
            }
            $message_type = 'success';
        } catch (Exception $e) {
            $message = 'Database error: ' . $e->getMessage();
            $message_type = 'error';
        }
    }
    
    // Handle delete service
    if (isset($_POST['delete_service'])) {
        try {
            $stmt = $db->prepare("DELETE FROM services WHERE id = ?");
            $stmt->execute([intval($_POST['service_id'])]);
            $message = 'Service deleted successfully!';
            $message_type = 'success';
        } catch (Exception $e) {
            $message = 'Error deleting service: ' . $e->getMessage();
            $message_type = 'error';
        }
    }
    
    // Handle toggle status
    if (isset($_POST['toggle_status'])) {
        try {
            $field = $_POST['field'];
            if (in_array($field, ['is_featured', 'is_active'])) {
                $stmt = $db->prepare("UPDATE services SET $field = ? WHERE id = ?");
                $stmt->execute([intval($_POST['value']), intval($_POST['service_id'])]);
                $message = 'Service status updated!';
                $message_type = 'success';
            }
        } catch (Exception $e) {
            $message = 'Error: ' . $e->getMessage();
            $message_type = 'error';
        }
    }
}

// Get all services
try {
    $db = getDB();
    $stmt = $db->query("SELECT * FROM services ORDER BY display_order, created_at DESC");
    $services = $stmt->fetchAll();
} catch (Exception $e) {
    $services = [];
}

// Get service for editing
$edit_service = null;
if (isset($_GET['edit']) && is_numeric($_GET['edit'])) {
    $edit_id = intval($_GET['edit']);
    foreach ($services as $service) {
        if ($service['id'] == $edit_id) {
            $edit_service = $service;
            break;
        }
    }
}

$page_title = 'Services Management';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $page_title; ?> - Admin Panel</title>
    <link rel="icon" type="image/x-icon" href="../public/favicon.ico">
    
    <!-- Google Fonts -->
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    
    <!-- Font Awesome -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    
    <style>
        :root {
            --primary-purple: #6B46C1;
            --primary-gold: #B8860B;
            --primary-maroon: #8B1538;
            --text-primary: #1F2937;
            --text-secondary: #6B7280;
            --bg-light: #F9FAFB;
            --bg-white: #FFFFFF;
            --border-color: #E5E7EB;
            --success-green: #10B981;
            --warning-yellow: #F59E0B;
            --error-red: #EF4444;
            --shadow-sm: 0 1px 2px 0 rgba(0, 0, 0, 0.05);
            --shadow-md: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
            --shadow-lg: 0 10px 15px -3px rgba(0, 0, 0, 0.1), 0 4px 6px -2px rgba(0, 0, 0, 0.05);
            --radius-md: 8px;
            --radius-lg: 12px;
            --transition: all 0.3s ease;
        }
        
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Poppins', sans-serif;
            background: var(--bg-light);
            color: var(--text-primary);
            line-height: 1.6;
        }
        
        .admin-layout {
            display: flex;
            min-height: 100vh;
        }
        
        .sidebar {
            width: 280px;
            background: var(--bg-white);
            border-right: 1px solid var(--border-color);
            padding: 2rem 0;
            position: fixed;
            height: 100vh;
            overflow-y: auto;
            box-shadow: var(--shadow-sm);
        }
        
        .sidebar-header {
            padding: 0 2rem 2rem;
            border-bottom: 1px solid var(--border-color);
            margin-bottom: 2rem;
        }
        
        .sidebar-logo {
            display: flex;
            align-items: center;
            gap: 1rem;
        }
        
        .sidebar-logo .logo-icon {
            width: 40px;
            height: 40px;
            background: var(--primary-gold);
            border-radius: 8px;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
        }
        
        .sidebar-title {
            font-size: 1.2rem;
            font-weight: 600;
            color: var(--text-primary);
        }
        
        .sidebar-subtitle {
            font-size: 0.8rem;
            color: var(--text-secondary);
        }
        
        .sidebar-nav {
            padding: 0 1rem;
        }
        
        .nav-item {
            margin-bottom: 0.5rem;
        }
        
        .nav-link {
            display: flex;
            align-items: center;
            gap: 1rem;
            padding: 1rem;
            color: var(--text-secondary);
            text-decoration: none;
            border-radius: var(--radius-md);
            transition: var(--transition);
            font-weight: 500;
        }
        
        .nav-link:hover {
            background: var(--bg-light);
            color: var(--primary-purple);
        }
        
        .nav-link.active {
            background: var(--primary-purple);
            color: white;
        }
        
        .nav-link i {
            width: 20px;
            text-align: center;
        }
        
        .main-content {
            flex: 1;
            margin-left: 280px;
            padding: 2rem;
        }
        
        .page-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 2rem;
            padding-bottom: 1rem;
            border-bottom: 1px solid var(--border-color);
        }
        
        .page-title {
            font-size: 2rem;
            font-weight: 600;
            color: var(--text-primary);
        }
        
        .admin-info {
            display: flex;
            align-items: center;
            gap: 1rem;
        }
        
        .admin-avatar {
            width: 40px;
            height: 40px;
            background: var(--primary-gold);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-weight: 600;
        }
        
        .admin-details {
            display: flex;
            flex-direction: column;
        }
        
        .admin-name {
            font-weight: 600;
            font-size: 0.9rem;
        }
        
        .admin-role {
            font-size: 0.8rem;
            color: var(--text-secondary);
        }
        
        .logout-btn {
            background: var(--error-red);
            color: white;
            border: none;
            padding: 0.5rem 1rem;
            border-radius: var(--radius-md);
            cursor: pointer;
            font-size: 0.85rem;
            transition: var(--transition);
        }
        
        .logout-btn:hover {
            background: #DC2626;
        }
        
        .alert {
            padding: 1rem 1.5rem;
            border-radius: var(--radius-md);
            margin-bottom: 2rem;
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }
        
        .alert-success {
            background: rgba(16, 185, 129, 0.1);
            border: 1px solid rgba(16, 185, 129, 0.2);
            color: var(--success-green);
        }
        
        .alert-error {
            background: rgba(239, 68, 68, 0.1);
            border: 1px solid rgba(239, 68, 68, 0.2);
            color: var(--error-red);
        }
        
        .form-section {
            background: var(--bg-white);
            padding: 2rem;
            border-radius: var(--radius-lg);
            box-shadow: var(--shadow-md);
            margin-bottom: 2rem;
        }
        
        .section-title {
            font-size: 1.3rem;
            font-weight: 600;
            color: var(--text-primary);
            margin-bottom: 1.5rem;
            padding-bottom: 0.5rem;
            border-bottom: 2px solid var(--primary-gold);
        }
        
        .form-grid {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 1.5rem;
            margin-bottom: 1.5rem;
        }
        
        .form-group {
            margin-bottom: 1.5rem;
        }
        
        .form-group.full-width {
            grid-column: 1 / -1;
        }
        
        .form-label {
            display: block;
            font-weight: 500;
            color: var(--text-primary);
            margin-bottom: 0.5rem;
            font-size: 0.9rem;
        }
        
        .form-input {
            width: 100%;
            padding: 0.75rem;
            border: 1px solid var(--border-color);
            border-radius: var(--radius-md);
            font-size: 1rem;
            transition: var(--transition);
        }
        
        .form-input:focus {
            outline: none;
            border-color: var(--primary-purple);
            box-shadow: 0 0 0 3px rgba(107, 70, 193, 0.1);
        }
        
        .form-textarea {
            min-height: 100px;
            resize: vertical;
        }
        
        .form-checkbox {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            margin-bottom: 1rem;
        }
        
        .form-checkbox input[type="checkbox"] {
            width: 18px;
            height: 18px;
        }
        
        .btn {
            padding: 0.75rem 1.5rem;
            border: none;
            border-radius: var(--radius-md);
            font-weight: 600;
            cursor: pointer;
            transition: var(--transition);
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
        }
        
        .btn-primary {
            background: var(--primary-purple);
            color: white;
        }
        
        .btn-primary:hover {
            background: #553C9A;
        }
        
        .btn-secondary {
            background: var(--text-secondary);
            color: white;
        }
        
        .btn-secondary:hover {
            background: #4B5563;
        }
        
        .btn-danger {
            background: var(--error-red);
            color: white;
        }
        
        .btn-danger:hover {
            background: #DC2626;
        }
        
        .btn-success {
            background: var(--success-green);
            color: white;
        }
        
        .btn-success:hover {
            background: #059669;
        }
        
        .btn-warning {
            background: var(--warning-yellow);
            color: white;
        }
        
        .btn-warning:hover {
            background: #D97706;
        }
        
        .btn-sm {
            padding: 0.5rem 0.75rem;
            font-size: 0.8rem;
        }
        
        .services-section {
            background: var(--bg-white);
            border-radius: var(--radius-lg);
            box-shadow: var(--shadow-md);
            overflow: hidden;
        }
        
        .services-header {
            padding: 1.5rem 2rem;
            background: var(--bg-light);
            border-bottom: 1px solid var(--border-color);
        }
        
        .services-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(350px, 1fr));
            gap: 1.5rem;
            padding: 2rem;
        }
        
        .service-card {
            border: 1px solid var(--border-color);
            border-radius: var(--radius-lg);
            overflow: hidden;
            transition: var(--transition);
        }
        
        .service-card:hover {
            box-shadow: var(--shadow-lg);
        }
        
        .service-content {
            padding: 1.5rem;
        }
        
        .service-title {
            font-size: 1.1rem;
            font-weight: 600;
            color: var(--text-primary);
            margin-bottom: 0.5rem;
        }
        
        .service-description {
            font-size: 0.85rem;
            color: var(--text-secondary);
            line-height: 1.6;
            margin-bottom: 1rem;
        }
        
        .service-meta {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 1rem;
            font-size: 0.8rem;
            color: var(--text-secondary);
        }
        
        .service-actions {
            display: flex;
            gap: 0.5rem;
            flex-wrap: wrap;
        }
        
        .status-badge {
            padding: 0.25rem 0.75rem;
            border-radius: 20px;
            font-size: 0.7rem;
            font-weight: 500;
            text-transform: uppercase;
        }
        
        .status-active {
            background: rgba(16, 185, 129, 0.1);
            color: var(--success-green);
        }
        
        .status-inactive {
            background: rgba(107, 114, 128, 0.1);
            color: var(--text-secondary);
        }
        
        .status-featured {
            background: rgba(245, 158, 11, 0.1);
            color: var(--warning-yellow);
        }
        
        .pricing-info {
            display: flex;
            gap: 1rem;
            margin: 0.5rem 0;
            font-size: 0.85rem;
        }
        
        .price-item {
            display: flex;
            align-items: center;
            gap: 0.25rem;
            color: var(--primary-gold);
            font-weight: 600;
        }
        
        .empty-state {
            padding: 3rem;
            text-align: center;
            color: var(--text-secondary);
        }
        
        .empty-state i {
            font-size: 3rem;
            margin-bottom: 1rem;
            opacity: 0.5;
        }
        
        @media (max-width: 768px) {
            .sidebar {
                width: 100%;
                height: auto;
                position: relative;
            }
            
            .main-content {
                margin-left: 0;
                padding: 1rem;
            }
            
            .form-grid {
                grid-template-columns: 1fr;
            }
            
            .services-grid {
                grid-template-columns: 1fr;
                padding: 1rem;
            }
            
            .page-header {
                flex-direction: column;
                gap: 1rem;
                align-items: flex-start;
            }
        }
    </style>
</head>
<body>
    <div class="admin-layout">
        <!-- Sidebar -->
        <div class="sidebar">
            <div class="sidebar-header">
                <div class="sidebar-logo">
                    <div class="logo-icon">
                        <i class="fas fa-crown"></i>
                    </div>
                    <div>
                        <div class="sidebar-title">SSV Admin</div>
                        <div class="sidebar-subtitle">Event Management</div>
                    </div>
                </div>
            </div>
            
            <nav class="sidebar-nav">
                <div class="nav-item">
                    <a href="dashboard.php" class="nav-link">
                        <i class="fas fa-tachometer-alt"></i>
                        Dashboard
                    </a>
                </div>
                <div class="nav-item">
                    <a href="hero-slides.php" class="nav-link">
                        <i class="fas fa-images"></i>
                        Hero Slider
                    </a>
                </div>
                <div class="nav-item">
                    <a href="services.php" class="nav-link active">
                        <i class="fas fa-concierge-bell"></i>
                        Services
                    </a>
                </div>
                <div class="nav-item">
                    <a href="gallery.php" class="nav-link">
                        <i class="fas fa-photo-video"></i>
                        Gallery
                    </a>
                </div>
                <div class="nav-item">
                    <a href="testimonials.php" class="nav-link">
                        <i class="fas fa-star"></i>
                        Testimonials
                    </a>
                </div>
                <div class="nav-item">
                    <a href="enquiries.php" class="nav-link">
                        <i class="fas fa-envelope"></i>
                        Enquiries
                    </a>
                </div>
                <div class="nav-item">
                    <a href="settings.php" class="nav-link">
                        <i class="fas fa-cog"></i>
                        Settings
                    </a>
                </div>
            </nav>
        </div>
        
        <!-- Main Content -->
        <div class="main-content">
            <div class="page-header">
                <h1 class="page-title"><?php echo $page_title; ?></h1>
                <div class="admin-info">
                    <div class="admin-avatar">
                        <?php echo strtoupper(substr($admin['name'], 0, 1)); ?>
                    </div>
                    <div class="admin-details">
                        <div class="admin-name"><?php echo htmlspecialchars($admin['name']); ?></div>
                        <div class="admin-role">Administrator</div>
                    </div>
                    <form method="POST" style="margin-left: 1rem;">
                        <button type="submit" name="logout" class="logout-btn">
                            <i class="fas fa-sign-out-alt"></i> Logout
                        </button>
                    </form>
                </div>
            </div>
            
            <!-- Alert Messages -->
            <?php if ($message): ?>
                <div class="alert alert-<?php echo $message_type; ?>">
                    <i class="fas fa-<?php echo $message_type === 'success' ? 'check-circle' : 'exclamation-triangle'; ?>"></i>
                    <?php echo htmlspecialchars($message); ?>
                </div>
            <?php endif; ?>
            
            <!-- Add/Edit Service Form -->
            <div class="form-section">
                <h2 class="section-title">
                    <?php echo $edit_service ? 'Edit Service' : 'Add New Service'; ?>
                </h2>
                
                <form method="POST">
                    <?php if ($edit_service): ?>
                        <input type="hidden" name="service_id" value="<?php echo $edit_service['id']; ?>">
                    <?php endif; ?>
                    
                    <div class="form-grid">
                        <div class="form-group">
                            <label for="title" class="form-label">Service Title *</label>
                            <input type="text" id="title" name="title" class="form-input" 
                                   value="<?php echo htmlspecialchars($edit_service['title'] ?? ''); ?>" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="display_order" class="form-label">Display Order</label>
                            <input type="number" id="display_order" name="display_order" class="form-input" 
                                   value="<?php echo htmlspecialchars($edit_service['display_order'] ?? '0'); ?>" min="0">
                        </div>
                        
                        <div class="form-group full-width">
                            <label for="short_description" class="form-label">Short Description</label>
                            <input type="text" id="short_description" name="short_description" class="form-input" 
                                   value="<?php echo htmlspecialchars($edit_service['short_description'] ?? ''); ?>"
                                   placeholder="Brief description for cards and previews">
                        </div>
                        
                        <div class="form-group full-width">
                            <label for="description" class="form-label">Full Description *</label>
                            <textarea id="description" name="description" class="form-input form-textarea" required><?php echo htmlspecialchars($edit_service['description'] ?? ''); ?></textarea>
                        </div>
                        
                        <div class="form-group">
                            <label for="capacity" class="form-label">Capacity (Guests)</label>
                            <input type="number" id="capacity" name="capacity" class="form-input" 
                                   value="<?php echo htmlspecialchars($edit_service['capacity'] ?? ''); ?>" min="0"
                                   placeholder="Maximum number of guests">
                        </div>
                        
                        <div class="form-group">
                            <label for="per_plate_charge" class="form-label">Per Plate Charge (₹)</label>
                            <input type="number" id="per_plate_charge" name="per_plate_charge" class="form-input" 
                                   value="<?php echo htmlspecialchars($edit_service['per_plate_charge'] ?? ''); ?>" min="0" step="0.01"
                                   placeholder="Price per plate in rupees">
                        </div>
                    </div>
                    
                    <div style="display: flex; gap: 2rem; margin-bottom: 2rem;">
                        <div class="form-checkbox">
                            <input type="checkbox" id="is_featured" name="is_featured" <?php echo ($edit_service['is_featured'] ?? 0) ? 'checked' : ''; ?>>
                            <label for="is_featured" class="form-label" style="margin-bottom: 0;">Featured Service</label>
                        </div>
                        
                        <div class="form-checkbox">
                            <input type="checkbox" id="is_active" name="is_active" <?php echo ($edit_service['is_active'] ?? 1) ? 'checked' : ''; ?>>
                            <label for="is_active" class="form-label" style="margin-bottom: 0;">Active</label>
                        </div>
                    </div>
                    
                    <div style="display: flex; gap: 1rem;">
                        <button type="submit" name="save_service" class="btn btn-primary">
                            <i class="fas fa-save"></i>
                            <?php echo $edit_service ? 'Update Service' : 'Add Service'; ?>
                        </button>
                        
                        <?php if ($edit_service): ?>
                            <a href="services.php" class="btn btn-secondary">
                                <i class="fas fa-times"></i>
                                Cancel
                            </a>
                        <?php endif; ?>
                    </div>
                </form>
            </div>
            
            <!-- Existing Services -->
            <div class="services-section">
                <div class="services-header">
                    <h2 class="section-title">Existing Services</h2>
                </div>
                
                <?php if (empty($services)): ?>
                    <div class="empty-state">
                        <i class="fas fa-concierge-bell"></i>
                        <h3>No services yet</h3>
                        <p>Add your first service using the form above.</p>
                    </div>
                <?php else: ?>
                    <div class="services-grid">
                        <?php foreach ($services as $service): ?>
                            <div class="service-card">
                                <div class="service-content">
                                    <h3 class="service-title"><?php echo htmlspecialchars($service['title']); ?></h3>
                                    
                                    <p class="service-description">
                                        <?php echo htmlspecialchars(substr($service['short_description'] ?: $service['description'], 0, 150)) . (strlen($service['short_description'] ?: $service['description']) > 150 ? '...' : ''); ?>
                                    </p>
                                    
                                    <?php if ($service['capacity'] || $service['per_plate_charge']): ?>
                                        <div class="pricing-info">
                                            <?php if ($service['capacity']): ?>
                                                <div class="price-item">
                                                    <i class="fas fa-users"></i>
                                                    <?php echo $service['capacity']; ?> guests
                                                </div>
                                            <?php endif; ?>
                                            <?php if ($service['per_plate_charge']): ?>
                                                <div class="price-item">
                                                    <i class="fas fa-rupee-sign"></i>
                                                    <?php echo number_format($service['per_plate_charge']); ?>/plate
                                                </div>
                                            <?php endif; ?>
                                        </div>
                                    <?php endif; ?>
                                    
                                    <div class="service-meta">
                                        <div>
                                            <span class="status-badge status-<?php echo $service['is_active'] ? 'active' : 'inactive'; ?>">
                                                <?php echo $service['is_active'] ? 'Active' : 'Inactive'; ?>
                                            </span>
                                            <?php if ($service['is_featured']): ?>
                                                <span class="status-badge status-featured">Featured</span>
                                            <?php endif; ?>
                                        </div>
                                        <div>Order: <?php echo $service['display_order']; ?></div>
                                    </div>
                                    
                                    <div class="service-actions">
                                        <a href="services.php?edit=<?php echo $service['id']; ?>" class="btn btn-secondary btn-sm">
                                            <i class="fas fa-edit"></i> Edit
                                        </a>
                                        
                                        <form method="POST" style="display: inline;" onsubmit="return confirm('Are you sure you want to delete this service?');">
                                            <input type="hidden" name="service_id" value="<?php echo $service['id']; ?>">
                                            <button type="submit" name="delete_service" class="btn btn-danger btn-sm">
                                                <i class="fas fa-trash"></i> Delete
                                            </button>
                                        </form>
                                        
                                        <form method="POST" style="display: inline;">
                                            <input type="hidden" name="service_id" value="<?php echo $service['id']; ?>">
                                            <input type="hidden" name="field" value="is_active">
                                            <input type="hidden" name="value" value="<?php echo $service['is_active'] ? 0 : 1; ?>">
                                            <button type="submit" name="toggle_status" class="btn btn-<?php echo $service['is_active'] ? 'secondary' : 'success'; ?> btn-sm">
                                                <i class="fas fa-<?php echo $service['is_active'] ? 'eye-slash' : 'eye'; ?>"></i>
                                                <?php echo $service['is_active'] ? 'Hide' : 'Show'; ?>
                                            </button>
                                        </form>
                                        
                                        <form method="POST" style="display: inline;">
                                            <input type="hidden" name="service_id" value="<?php echo $service['id']; ?>">
                                            <input type="hidden" name="field" value="is_featured">
                                            <input type="hidden" name="value" value="<?php echo $service['is_featured'] ? 0 : 1; ?>">
                                            <button type="submit" name="toggle_status" class="btn btn-<?php echo $service['is_featured'] ? 'secondary' : 'warning'; ?> btn-sm">
                                                <i class="fas fa-star"></i>
                                                <?php echo $service['is_featured'] ? 'Unfeature' : 'Feature'; ?>
                                            </button>
                                        </form>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</body>
</html>