<?php
session_start();

require_once '../config/db.php';
require_once 'auth.php';
requireAdminLogin();

$admin = getCurrentAdmin();
$message = '';
$message_type = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $db = getDB();
    
    if (isset($_POST['logout'])) {
        adminLogout();
    }
    
    if (isset($_POST['save_gallery'])) {
        $gallery_id = $_POST['gallery_id'] ?? 0;
        $title = sanitizeInput($_POST['title'] ?? '');
        $description = sanitizeInput($_POST['description'] ?? '');
        $service_category = sanitizeInput($_POST['service_category'] ?? '');
        $alt_text = sanitizeInput($_POST['alt_text'] ?? '');
        $is_featured = isset($_POST['is_featured']) ? 1 : 0;
        $display_order = intval($_POST['display_order'] ?? 0);
        
        // Handle file upload
        if (isset($_FILES['media_file']) && $_FILES['media_file']['error'] !== UPLOAD_ERR_NO_FILE) {
            $upload_dir = '../public/assets/images/gallery/';
            $upload_result = handleFileUpload($_FILES['media_file'], $upload_dir, ['jpg', 'jpeg', 'png', 'webp', 'mp4', 'mov'], 20971520); // 20MB limit
            
            if ($upload_result['success']) {
                $file_name = $upload_result['filename'];
                $file_type = in_array(pathinfo($file_name, PATHINFO_EXTENSION), ['mp4', 'mov']) ? 'video' : 'image';
                $file_size = $_FILES['media_file']['size'];
                
                try {
                    if ($gallery_id > 0) {
                        // Delete old file
                        $stmt = $db->prepare("SELECT file_name FROM gallery WHERE id = ?");
                        $stmt->execute([$gallery_id]);
                        $old_file = $stmt->fetch();
                        if ($old_file && file_exists($upload_dir . $old_file['file_name'])) {
                            unlink($upload_dir . $old_file['file_name']);
                        }
                        
                        $stmt = $db->prepare("UPDATE gallery SET title = ?, description = ?, file_name = ?, file_type = ?, file_size = ?, service_category = ?, alt_text = ?, is_featured = ?, display_order = ? WHERE id = ?");
                        $stmt->execute([$title, $description, $file_name, $file_type, $file_size, $service_category, $alt_text, $is_featured, $display_order, $gallery_id]);
                        $message = 'Gallery item updated successfully!';
                    } else {
                        $stmt = $db->prepare("INSERT INTO gallery (title, description, file_name, file_type, file_size, service_category, alt_text, is_featured, display_order) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)");
                        $stmt->execute([$title, $description, $file_name, $file_type, $file_size, $service_category, $alt_text, $is_featured, $display_order]);
                        $message = 'Gallery item added successfully!';
                    }
                    $message_type = 'success';
                } catch (Exception $e) {
                    $message = 'Database error: ' . $e->getMessage();
                    $message_type = 'error';
                }
            } else {
                $message = $upload_result['message'];
                $message_type = 'error';
            }
        } else if ($gallery_id == 0) {
            $message = 'Please select a file to upload.';
            $message_type = 'error';
        }
    }
    
    if (isset($_POST['delete_gallery'])) {
        try {
            $stmt = $db->prepare("SELECT file_name FROM gallery WHERE id = ?");
            $stmt->execute([intval($_POST['gallery_id'])]);
            $item = $stmt->fetch();
            
            if ($item) {
                $stmt = $db->prepare("DELETE FROM gallery WHERE id = ?");
                $stmt->execute([intval($_POST['gallery_id'])]);
                
                if (file_exists('../public/assets/images/gallery/' . $item['file_name'])) {
                    unlink('../public/assets/images/gallery/' . $item['file_name']);
                }
                
                $message = 'Gallery item deleted successfully!';
                $message_type = 'success';
            }
        } catch (Exception $e) {
            $message = 'Error: ' . $e->getMessage();
            $message_type = 'error';
        }
    }
    
    if (isset($_POST['toggle_featured'])) {
        try {
            $stmt = $db->prepare("UPDATE gallery SET is_featured = ? WHERE id = ?");
            $stmt->execute([intval($_POST['value']), intval($_POST['gallery_id'])]);
            $message = 'Gallery item updated!';
            $message_type = 'success';
        } catch (Exception $e) {
            $message = 'Error: ' . $e->getMessage();
            $message_type = 'error';
        }
    }
}

// Get gallery items
try {
    $db = getDB();
    $stmt = $db->query("SELECT * FROM gallery ORDER BY display_order, created_at DESC");
    $gallery_items = $stmt->fetchAll();
} catch (Exception $e) {
    $gallery_items = [];
}

// Get categories
$categories = ['Wedding', 'Engagement', 'Birthday', 'Corporate', 'Reception', 'Anniversary'];

$edit_item = null;
if (isset($_GET['edit']) && is_numeric($_GET['edit'])) {
    foreach ($gallery_items as $item) {
        if ($item['id'] == intval($_GET['edit'])) {
            $edit_item = $item;
            break;
        }
    }
}

$page_title = 'Gallery Management';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $page_title; ?> - Admin Panel</title>
    <link rel="icon" type="image/x-icon" href="../public/favicon.ico">
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --primary-purple: #6B46C1;
            --primary-gold: #B8860B;
            --text-primary: #1F2937;
            --text-secondary: #6B7280;
            --bg-light: #F9FAFB;
            --bg-white: #FFFFFF;
            --border-color: #E5E7EB;
            --success-green: #10B981;
            --warning-yellow: #F59E0B;
            --error-red: #EF4444;
            --shadow-md: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
            --radius-md: 8px;
            --radius-lg: 12px;
            --transition: all 0.3s ease;
        }
        
        * { margin: 0; padding: 0; box-sizing: border-box; }
        
        body {
            font-family: 'Poppins', sans-serif;
            background: var(--bg-light);
            color: var(--text-primary);
            line-height: 1.6;
        }
        
        .admin-layout { display: flex; min-height: 100vh; }
        
        .sidebar {
            width: 280px;
            background: var(--bg-white);
            border-right: 1px solid var(--border-color);
            padding: 2rem 0;
            position: fixed;
            height: 100vh;
            overflow-y: auto;
            box-shadow: 0 1px 2px 0 rgba(0, 0, 0, 0.05);
        }
        
        .sidebar-header {
            padding: 0 2rem 2rem;
            border-bottom: 1px solid var(--border-color);
            margin-bottom: 2rem;
        }
        
        .sidebar-logo {
            display: flex;
            align-items: center;
            gap: 1rem;
        }
        
        .sidebar-logo .logo-icon {
            width: 40px;
            height: 40px;
            background: var(--primary-gold);
            border-radius: 8px;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
        }
        
        .sidebar-title {
            font-size: 1.2rem;
            font-weight: 600;
            color: var(--text-primary);
        }
        
        .sidebar-subtitle {
            font-size: 0.8rem;
            color: var(--text-secondary);
        }
        
        .sidebar-nav { padding: 0 1rem; }
        
        .nav-item { margin-bottom: 0.5rem; }
        
        .nav-link {
            display: flex;
            align-items: center;
            gap: 1rem;
            padding: 1rem;
            color: var(--text-secondary);
            text-decoration: none;
            border-radius: var(--radius-md);
            transition: var(--transition);
            font-weight: 500;
        }
        
        .nav-link:hover {
            background: var(--bg-light);
            color: var(--primary-purple);
        }
        
        .nav-link.active {
            background: var(--primary-purple);
            color: white;
        }
        
        .nav-link i { width: 20px; text-align: center; }
        
        .main-content {
            flex: 1;
            margin-left: 280px;
            padding: 2rem;
        }
        
        .page-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 2rem;
            padding-bottom: 1rem;
            border-bottom: 1px solid var(--border-color);
        }
        
        .page-title {
            font-size: 2rem;
            font-weight: 600;
            color: var(--text-primary);
        }
        
        .admin-info {
            display: flex;
            align-items: center;
            gap: 1rem;
        }
        
        .admin-avatar {
            width: 40px;
            height: 40px;
            background: var(--primary-gold);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-weight: 600;
        }
        
        .admin-details {
            display: flex;
            flex-direction: column;
        }
        
        .admin-name {
            font-weight: 600;
            font-size: 0.9rem;
        }
        
        .admin-role {
            font-size: 0.8rem;
            color: var(--text-secondary);
        }
        
        .logout-btn {
            background: var(--error-red);
            color: white;
            border: none;
            padding: 0.5rem 1rem;
            border-radius: var(--radius-md);
            cursor: pointer;
            font-size: 0.85rem;
            transition: var(--transition);
        }
        
        .logout-btn:hover { background: #DC2626; }
        
        .alert {
            padding: 1rem 1.5rem;
            border-radius: var(--radius-md);
            margin-bottom: 2rem;
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }
        
        .alert-success {
            background: rgba(16, 185, 129, 0.1);
            border: 1px solid rgba(16, 185, 129, 0.2);
            color: var(--success-green);
        }
        
        .alert-error {
            background: rgba(239, 68, 68, 0.1);
            border: 1px solid rgba(239, 68, 68, 0.2);
            color: var(--error-red);
        }
        
        .form-section {
            background: var(--bg-white);
            padding: 2rem;
            border-radius: var(--radius-lg);
            box-shadow: var(--shadow-md);
            margin-bottom: 2rem;
        }
        
        .section-title {
            font-size: 1.3rem;
            font-weight: 600;
            color: var(--text-primary);
            margin-bottom: 1.5rem;
            padding-bottom: 0.5rem;
            border-bottom: 2px solid var(--primary-gold);
        }
        
        .form-grid {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 1.5rem;
            margin-bottom: 1.5rem;
        }
        
        .form-group { margin-bottom: 1.5rem; }
        
        .form-group.full-width { grid-column: 1 / -1; }
        
        .form-label {
            display: block;
            font-weight: 500;
            color: var(--text-primary);
            margin-bottom: 0.5rem;
            font-size: 0.9rem;
        }
        
        .form-input {
            width: 100%;
            padding: 0.75rem;
            border: 1px solid var(--border-color);
            border-radius: var(--radius-md);
            font-size: 1rem;
            transition: var(--transition);
        }
        
        .form-input:focus {
            outline: none;
            border-color: var(--primary-purple);
            box-shadow: 0 0 0 3px rgba(107, 70, 193, 0.1);
        }
        
        .form-textarea { min-height: 100px; resize: vertical; }
        
        .form-checkbox {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            margin-bottom: 1rem;
        }
        
        .form-checkbox input[type="checkbox"] {
            width: 18px;
            height: 18px;
        }
        
        .btn {
            padding: 0.75rem 1.5rem;
            border: none;
            border-radius: var(--radius-md);
            font-weight: 600;
            cursor: pointer;
            transition: var(--transition);
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
        }
        
        .btn-sm { padding: 0.5rem 0.75rem; font-size: 0.8rem; }
        .btn-primary { background: var(--primary-purple); color: white; }
        .btn-primary:hover { background: #553C9A; }
        .btn-secondary { background: var(--text-secondary); color: white; }
        .btn-secondary:hover { background: #4B5563; }
        .btn-danger { background: var(--error-red); color: white; }
        .btn-danger:hover { background: #DC2626; }
        .btn-warning { background: var(--warning-yellow); color: white; }
        .btn-warning:hover { background: #D97706; }
        
        .gallery-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
            gap: 1.5rem;
            padding: 2rem;
        }
        
        .gallery-card {
            border: 1px solid var(--border-color);
            border-radius: var(--radius-lg);
            overflow: hidden;
            transition: var(--transition);
        }
        
        .gallery-card:hover { box-shadow: var(--shadow-md); }
        
        .gallery-media {
            width: 100%;
            height: 200px;
            object-fit: cover;
            background: var(--bg-light);
        }
        
        .gallery-content { padding: 1.5rem; }
        
        .gallery-title {
            font-size: 1.1rem;
            font-weight: 600;
            color: var(--text-primary);
            margin-bottom: 0.5rem;
        }
        
        .gallery-meta {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin: 1rem 0;
            font-size: 0.8rem;
            color: var(--text-secondary);
        }
        
        .gallery-actions {
            display: flex;
            gap: 0.5rem;
            flex-wrap: wrap;
        }
        
        .status-badge {
            padding: 0.25rem 0.75rem;
            border-radius: 20px;
            font-size: 0.7rem;
            font-weight: 500;
            text-transform: uppercase;
        }
        
        .status-featured {
            background: rgba(245, 158, 11, 0.1);
            color: var(--warning-yellow);
        }
        
        .file-info {
            display: flex;
            gap: 1rem;
            margin: 0.5rem 0;
            font-size: 0.85rem;
            color: var(--text-secondary);
        }
        
        .empty-state {
            padding: 3rem;
            text-align: center;
            color: var(--text-secondary);
        }
        
        .empty-state i {
            font-size: 3rem;
            margin-bottom: 1rem;
            opacity: 0.5;
        }
        
        @media (max-width: 768px) {
            .sidebar { width: 100%; height: auto; position: relative; }
            .main-content { margin-left: 0; padding: 1rem; }
            .form-grid { grid-template-columns: 1fr; }
            .gallery-grid { grid-template-columns: 1fr; padding: 1rem; }
            .page-header { flex-direction: column; gap: 1rem; align-items: flex-start; }
        }
    </style>
</head>
<body>
    <div class="admin-layout">
        <!-- Sidebar -->
        <div class="sidebar">
            <div class="sidebar-header">
                <div class="sidebar-logo">
                    <div class="logo-icon">
                        <i class="fas fa-crown"></i>
                    </div>
                    <div>
                        <div class="sidebar-title">SSV Admin</div>
                        <div class="sidebar-subtitle">Event Management</div>
                    </div>
                </div>
            </div>
            
            <nav class="sidebar-nav">
                <div class="nav-item">
                    <a href="dashboard.php" class="nav-link">
                        <i class="fas fa-tachometer-alt"></i>
                        Dashboard
                    </a>
                </div>
                <div class="nav-item">
                    <a href="hero-slides.php" class="nav-link">
                        <i class="fas fa-images"></i>
                        Hero Slider
                    </a>
                </div>
                <div class="nav-item">
                    <a href="services.php" class="nav-link">
                        <i class="fas fa-concierge-bell"></i>
                        Services
                    </a>
                </div>
                <div class="nav-item">
                    <a href="gallery.php" class="nav-link active">
                        <i class="fas fa-photo-video"></i>
                        Gallery
                    </a>
                </div>
                <div class="nav-item">
                    <a href="testimonials.php" class="nav-link">
                        <i class="fas fa-star"></i>
                        Testimonials
                    </a>
                </div>
                <div class="nav-item">
                    <a href="enquiries.php" class="nav-link">
                        <i class="fas fa-envelope"></i>
                        Enquiries
                    </a>
                </div>
                <div class="nav-item">
                    <a href="settings.php" class="nav-link">
                        <i class="fas fa-cog"></i>
                        Settings
                    </a>
                </div>
            </nav>
        </div>
        
        <!-- Main Content -->
        <div class="main-content">
            <div class="page-header">
                <h1 class="page-title"><?php echo $page_title; ?></h1>
                <div class="admin-info">
                    <div class="admin-avatar">
                        <?php echo strtoupper(substr($admin['name'], 0, 1)); ?>
                    </div>
                    <div class="admin-details">
                        <div class="admin-name"><?php echo htmlspecialchars($admin['name']); ?></div>
                        <div class="admin-role">Administrator</div>
                    </div>
                    <form method="POST" style="margin-left: 1rem;">
                        <button type="submit" name="logout" class="logout-btn">
                            <i class="fas fa-sign-out-alt"></i> Logout
                        </button>
                    </form>
                </div>
            </div>
            
            <?php if ($message): ?>
                <div class="alert alert-<?php echo $message_type; ?>">
                    <i class="fas fa-<?php echo $message_type === 'success' ? 'check-circle' : 'exclamation-triangle'; ?>"></i>
                    <?php echo htmlspecialchars($message); ?>
                </div>
            <?php endif; ?>
            
            <!-- Add/Edit Form -->
            <div class="form-section">
                <h2 class="section-title">
                    <?php echo $edit_item ? 'Edit Gallery Item' : 'Add New Gallery Item'; ?>
                </h2>
                
                <form method="POST" enctype="multipart/form-data">
                    <?php if ($edit_item): ?>
                        <input type="hidden" name="gallery_id" value="<?php echo $edit_item['id']; ?>">
                    <?php endif; ?>
                    
                    <div class="form-grid">
                        <div class="form-group">
                            <label for="title" class="form-label">Title</label>
                            <input type="text" id="title" name="title" class="form-input" 
                                   value="<?php echo htmlspecialchars($edit_item['title'] ?? ''); ?>">
                        </div>
                        
                        <div class="form-group">
                            <label for="service_category" class="form-label">Category</label>
                            <select id="service_category" name="service_category" class="form-input">
                                <option value="">Select Category</option>
                                <?php foreach ($categories as $cat): ?>
                                    <option value="<?php echo $cat; ?>" <?php echo ($edit_item['service_category'] ?? '') === $cat ? 'selected' : ''; ?>>
                                        <?php echo $cat; ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div class="form-group full-width">
                            <label for="description" class="form-label">Description</label>
                            <textarea id="description" name="description" class="form-input form-textarea"><?php echo htmlspecialchars($edit_item['description'] ?? ''); ?></textarea>
                        </div>
                        
                        <div class="form-group">
                            <label for="media_file" class="form-label">Media File <?php echo $edit_item ? '' : '*'; ?></label>
                            <input type="file" id="media_file" name="media_file" class="form-input" 
                                   accept="image/*,video/*" <?php echo $edit_item ? '' : 'required'; ?>>
                            <?php if ($edit_item): ?>
                                <small style="color: var(--text-secondary); margin-top: 0.5rem; display: block;">
                                    Current: <?php echo htmlspecialchars($edit_item['file_name']); ?>
                                </small>
                            <?php endif; ?>
                        </div>
                        
                        <div class="form-group">
                            <label for="alt_text" class="form-label">Alt Text</label>
                            <input type="text" id="alt_text" name="alt_text" class="form-input" 
                                   value="<?php echo htmlspecialchars($edit_item['alt_text'] ?? ''); ?>"
                                   placeholder="Description for accessibility">
                        </div>
                        
                        <div class="form-group">
                            <label for="display_order" class="form-label">Display Order</label>
                            <input type="number" id="display_order" name="display_order" class="form-input" 
                                   value="<?php echo htmlspecialchars($edit_item['display_order'] ?? '0'); ?>" min="0">
                        </div>
                    </div>
                    
                    <div class="form-checkbox">
                        <input type="checkbox" id="is_featured" name="is_featured" <?php echo ($edit_item['is_featured'] ?? 0) ? 'checked' : ''; ?>>
                        <label for="is_featured" class="form-label" style="margin-bottom: 0;">Featured Item</label>
                    </div>
                    
                    <div style="display: flex; gap: 1rem;">
                        <button type="submit" name="save_gallery" class="btn btn-primary">
                            <i class="fas fa-save"></i>
                            <?php echo $edit_item ? 'Update Item' : 'Add Item'; ?>
                        </button>
                        
                        <?php if ($edit_item): ?>
                            <a href="gallery.php" class="btn btn-secondary">
                                <i class="fas fa-times"></i>
                                Cancel
                            </a>
                        <?php endif; ?>
                    </div>
                </form>
            </div>
            
            <!-- Gallery Items -->
            <div class="form-section">
                <h2 class="section-title">Gallery Items</h2>
                
                <?php if (empty($gallery_items)): ?>
                    <div class="empty-state">
                        <i class="fas fa-photo-video"></i>
                        <h3>No gallery items yet</h3>
                        <p>Add your first gallery item using the form above.</p>
                    </div>
                <?php else: ?>
                    <div class="gallery-grid">
                        <?php foreach ($gallery_items as $item): ?>
                            <div class="gallery-card">
                                <?php if ($item['file_type'] === 'video'): ?>
                                    <video class="gallery-media" controls>
                                        <source src="../public/assets/images/gallery/<?php echo htmlspecialchars($item['file_name']); ?>" type="video/mp4">
                                    </video>
                                <?php else: ?>
                                    <img src="../public/assets/images/gallery/<?php echo htmlspecialchars($item['file_name']); ?>" 
                                         alt="<?php echo htmlspecialchars($item['alt_text'] ?: $item['title']); ?>" 
                                         class="gallery-media">
                                <?php endif; ?>
                                
                                <div class="gallery-content">
                                    <h3 class="gallery-title"><?php echo htmlspecialchars($item['title'] ?: 'Untitled'); ?></h3>
                                    
                                    <div class="file-info">
                                        <span><i class="fas fa-tag"></i> <?php echo ucfirst($item['file_type']); ?></span>
                                        <?php if ($item['service_category']): ?>
                                            <span><i class="fas fa-folder"></i> <?php echo htmlspecialchars($item['service_category']); ?></span>
                                        <?php endif; ?>
                                        <span><i class="fas fa-weight-hanging"></i> <?php echo formatFileSize($item['file_size']); ?></span>
                                    </div>
                                    
                                    <div class="gallery-meta">
                                        <div>
                                            <?php if ($item['is_featured']): ?>
                                                <span class="status-badge status-featured">Featured</span>
                                            <?php endif; ?>
                                        </div>
                                        <div>Order: <?php echo $item['display_order']; ?></div>
                                    </div>
                                    
                                    <div class="gallery-actions">
                                        <a href="gallery.php?edit=<?php echo $item['id']; ?>" class="btn btn-secondary btn-sm">
                                            <i class="fas fa-edit"></i> Edit
                                        </a>
                                        
                                        <form method="POST" style="display: inline;" onsubmit="return confirm('Delete this item?');">
                                            <input type="hidden" name="gallery_id" value="<?php echo $item['id']; ?>">
                                            <button type="submit" name="delete_gallery" class="btn btn-danger btn-sm">
                                                <i class="fas fa-trash"></i> Delete
                                            </button>
                                        </form>
                                        
                                        <form method="POST" style="display: inline;">
                                            <input type="hidden" name="gallery_id" value="<?php echo $item['id']; ?>">
                                            <input type="hidden" name="value" value="<?php echo $item['is_featured'] ? 0 : 1; ?>">
                                            <button type="submit" name="toggle_featured" class="btn btn-warning btn-sm">
                                                <i class="fas fa-star"></i>
                                                <?php echo $item['is_featured'] ? 'Unfeature' : 'Feature'; ?>
                                            </button>
                                        </form>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</body>
</html>