<?php
session_start();

require_once '../config/db.php';
require_once 'auth.php';
requireAdminLogin();

$admin = getCurrentAdmin();
$message = '';
$message_type = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $db = getDB();
    
    if (isset($_POST['logout'])) {
        adminLogout();
    }
    
    if (isset($_POST['update_status'])) {
        try {
            $stmt = $db->prepare("UPDATE enquiries SET status = ?, updated_at = NOW() WHERE id = ?");
            $stmt->execute([sanitizeInput($_POST['new_status']), intval($_POST['enquiry_id'])]);
            $message = 'Status updated successfully!';
            $message_type = 'success';
        } catch (Exception $e) {
            $message = 'Error: ' . $e->getMessage();
            $message_type = 'error';
        }
    }
    
    if (isset($_POST['add_note'])) {
        try {
            $note = sanitizeInput($_POST['admin_note']);
            if (!empty($note)) {
                $stmt = $db->prepare("UPDATE enquiries SET admin_notes = ?, updated_at = NOW() WHERE id = ?");
                $stmt->execute([$note, intval($_POST['enquiry_id'])]);
                $message = 'Note added successfully!';
                $message_type = 'success';
            } else {
                $message = 'Please enter a note.';
                $message_type = 'error';
            }
        } catch (Exception $e) {
            $message = 'Error: ' . $e->getMessage();
            $message_type = 'error';
        }
    }
    
    if (isset($_POST['send_response'])) {
        $enquiry_id = intval($_POST['enquiry_id']);
        $response = sanitizeInput($_POST['response_message']);
        $update_status = isset($_POST['mark_resolved']) ? 1 : 0;
        
        try {
            // Get enquiry details for email
            $stmt = $db->prepare("SELECT * FROM enquiries WHERE id = ?");
            $stmt->execute([$enquiry_id]);
            $enquiry = $stmt->fetch();
            
            if ($enquiry && !empty($response)) {
                // Update enquiry with response
                $new_status = $update_status ? 'resolved' : $enquiry['status'];
                $stmt = $db->prepare("UPDATE enquiries SET admin_response = ?, status = ?, updated_at = NOW() WHERE id = ?");
                $stmt->execute([$response, $new_status, $enquiry_id]);
                
                // Here you would typically send an email to the customer
                // For now, we'll just show a success message
                $message = 'Response saved successfully!' . ($update_status ? ' Enquiry marked as resolved.' : '');
                $message_type = 'success';
            } else {
                $message = 'Please enter a response message.';
                $message_type = 'error';
            }
        } catch (Exception $e) {
            $message = 'Error: ' . $e->getMessage();
            $message_type = 'error';
        }
    }
    
    if (isset($_POST['delete_enquiry'])) {
        try {
            $stmt = $db->prepare("DELETE FROM enquiries WHERE id = ?");
            $stmt->execute([intval($_POST['enquiry_id'])]);
            $message = 'Enquiry deleted successfully!';
            $message_type = 'success';
        } catch (Exception $e) {
            $message = 'Error: ' . $e->getMessage();
            $message_type = 'error';
        }
    }
}

// Get enquiries
try {
    $db = getDB();
    $filter = $_GET['filter'] ?? 'all';
    $search = $_GET['search'] ?? '';
    
    $query = "SELECT * FROM enquiries WHERE 1=1";
    $params = [];
    
    if ($filter !== 'all') {
        $query .= " AND status = ?";
        $params[] = $filter;
    }
    
    if (!empty($search)) {
        $query .= " AND (name LIKE ? OR email LIKE ? OR phone LIKE ? OR message LIKE ?)";
        $search_term = '%' . $search . '%';
        $params[] = $search_term;
        $params[] = $search_term;
        $params[] = $search_term;
        $params[] = $search_term;
    }
    
    $query .= " ORDER BY created_at DESC";
    
    $stmt = $db->prepare($query);
    $stmt->execute($params);
    $enquiries = $stmt->fetchAll();
    
    // Get counts for filters
    $stmt = $db->query("SELECT status, COUNT(*) as count FROM enquiries GROUP BY status");
    $status_counts = [];
    while ($row = $stmt->fetch()) {
        $status_counts[$row['status']] = $row['count'];
    }
    $status_counts['all'] = array_sum($status_counts);
    
} catch (Exception $e) {
    $enquiries = [];
    $status_counts = ['all' => 0];
}

$statuses = ['new', 'in_progress', 'resolved', 'closed'];
$status_labels = [
    'new' => 'New',
    'in_progress' => 'In Progress', 
    'resolved' => 'Resolved',
    'closed' => 'Closed'
];

$view_enquiry = null;
if (isset($_GET['view']) && is_numeric($_GET['view'])) {
    foreach ($enquiries as $enquiry) {
        if ($enquiry['id'] == intval($_GET['view'])) {
            $view_enquiry = $enquiry;
            break;
        }
    }
}

$page_title = 'Enquiries Management';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $page_title; ?> - Admin Panel</title>
    <link rel="icon" type="image/x-icon" href="../public/favicon.ico">
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --primary-purple: #6B46C1;
            --primary-gold: #B8860B;
            --text-primary: #1F2937;
            --text-secondary: #6B7280;
            --bg-light: #F9FAFB;
            --bg-white: #FFFFFF;
            --border-color: #E5E7EB;
            --success-green: #10B981;
            --warning-yellow: #F59E0B;
            --error-red: #EF4444;
            --info-blue: #3B82F6;
            --shadow-md: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
            --radius-md: 8px;
            --radius-lg: 12px;
            --transition: all 0.3s ease;
        }
        
        * { margin: 0; padding: 0; box-sizing: border-box; }
        
        body {
            font-family: 'Poppins', sans-serif;
            background: var(--bg-light);
            color: var(--text-primary);
            line-height: 1.6;
        }
        
        .admin-layout { display: flex; min-height: 100vh; }
        
        .sidebar {
            width: 280px;
            background: var(--bg-white);
            border-right: 1px solid var(--border-color);
            padding: 2rem 0;
            position: fixed;
            height: 100vh;
            overflow-y: auto;
            box-shadow: 0 1px 2px 0 rgba(0, 0, 0, 0.05);
        }
        
        .sidebar-header {
            padding: 0 2rem 2rem;
            border-bottom: 1px solid var(--border-color);
            margin-bottom: 2rem;
        }
        
        .sidebar-logo {
            display: flex;
            align-items: center;
            gap: 1rem;
        }
        
        .sidebar-logo .logo-icon {
            width: 40px;
            height: 40px;
            background: var(--primary-gold);
            border-radius: 8px;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
        }
        
        .sidebar-title {
            font-size: 1.2rem;
            font-weight: 600;
            color: var(--text-primary);
        }
        
        .sidebar-subtitle {
            font-size: 0.8rem;
            color: var(--text-secondary);
        }
        
        .sidebar-nav { padding: 0 1rem; }
        
        .nav-item { margin-bottom: 0.5rem; }
        
        .nav-link {
            display: flex;
            align-items: center;
            gap: 1rem;
            padding: 1rem;
            color: var(--text-secondary);
            text-decoration: none;
            border-radius: var(--radius-md);
            transition: var(--transition);
            font-weight: 500;
        }
        
        .nav-link:hover {
            background: var(--bg-light);
            color: var(--primary-purple);
        }
        
        .nav-link.active {
            background: var(--primary-purple);
            color: white;
        }
        
        .nav-link i { width: 20px; text-align: center; }
        
        .main-content {
            flex: 1;
            margin-left: 280px;
            padding: 2rem;
        }
        
        .page-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 2rem;
            padding-bottom: 1rem;
            border-bottom: 1px solid var(--border-color);
        }
        
        .page-title {
            font-size: 2rem;
            font-weight: 600;
            color: var(--text-primary);
        }
        
        .admin-info {
            display: flex;
            align-items: center;
            gap: 1rem;
        }
        
        .admin-avatar {
            width: 40px;
            height: 40px;
            background: var(--primary-gold);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-weight: 600;
        }
        
        .admin-details {
            display: flex;
            flex-direction: column;
        }
        
        .admin-name {
            font-weight: 600;
            font-size: 0.9rem;
        }
        
        .admin-role {
            font-size: 0.8rem;
            color: var(--text-secondary);
        }
        
        .logout-btn {
            background: var(--error-red);
            color: white;
            border: none;
            padding: 0.5rem 1rem;
            border-radius: var(--radius-md);
            cursor: pointer;
            font-size: 0.85rem;
            transition: var(--transition);
        }
        
        .logout-btn:hover { background: #DC2626; }
        
        .alert {
            padding: 1rem 1.5rem;
            border-radius: var(--radius-md);
            margin-bottom: 2rem;
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }
        
        .alert-success {
            background: rgba(16, 185, 129, 0.1);
            border: 1px solid rgba(16, 185, 129, 0.2);
            color: var(--success-green);
        }
        
        .alert-error {
            background: rgba(239, 68, 68, 0.1);
            border: 1px solid rgba(239, 68, 68, 0.2);
            color: var(--error-red);
        }
        
        .form-section {
            background: var(--bg-white);
            padding: 2rem;
            border-radius: var(--radius-lg);
            box-shadow: var(--shadow-md);
            margin-bottom: 2rem;
        }
        
        .section-title {
            font-size: 1.3rem;
            font-weight: 600;
            color: var(--text-primary);
            margin-bottom: 1.5rem;
            padding-bottom: 0.5rem;
            border-bottom: 2px solid var(--primary-gold);
        }
        
        .btn {
            padding: 0.75rem 1.5rem;
            border: none;
            border-radius: var(--radius-md);
            font-weight: 600;
            cursor: pointer;
            transition: var(--transition);
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
        }
        
        .btn-sm { padding: 0.5rem 0.75rem; font-size: 0.8rem; }
        .btn-primary { background: var(--primary-purple); color: white; }
        .btn-primary:hover { background: #553C9A; }
        .btn-secondary { background: var(--text-secondary); color: white; }
        .btn-secondary:hover { background: #4B5563; }
        .btn-danger { background: var(--error-red); color: white; }
        .btn-danger:hover { background: #DC2626; }
        .btn-success { background: var(--success-green); color: white; }
        .btn-success:hover { background: #059669; }
        .btn-warning { background: var(--warning-yellow); color: white; }
        .btn-warning:hover { background: #D97706; }
        .btn-info { background: var(--info-blue); color: white; }
        .btn-info:hover { background: #2563EB; }
        
        .filters-section {
            display: flex;
            gap: 1rem;
            margin-bottom: 2rem;
            flex-wrap: wrap;
            align-items: center;
        }
        
        .filter-tabs {
            display: flex;
            gap: 0.5rem;
        }
        
        .filter-tab {
            padding: 0.5rem 1rem;
            border: 1px solid var(--border-color);
            border-radius: var(--radius-md);
            background: var(--bg-white);
            color: var(--text-secondary);
            text-decoration: none;
            font-size: 0.9rem;
            transition: var(--transition);
        }
        
        .filter-tab.active {
            background: var(--primary-purple);
            color: white;
            border-color: var(--primary-purple);
        }
        
        .search-form {
            display: flex;
            gap: 0.5rem;
        }
        
        .search-input {
            padding: 0.5rem 0.75rem;
            border: 1px solid var(--border-color);
            border-radius: var(--radius-md);
            font-size: 0.9rem;
            width: 250px;
        }
        
        .enquiry-grid {
            display: grid;
            grid-template-columns: 1fr;
            gap: 1.5rem;
        }
        
        .enquiry-card {
            border: 1px solid var(--border-color);
            border-radius: var(--radius-lg);
            padding: 1.5rem;
            background: var(--bg-white);
            transition: var(--transition);
        }
        
        .enquiry-card:hover { box-shadow: var(--shadow-md); }
        
        .enquiry-header {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            margin-bottom: 1rem;
        }
        
        .customer-info h3 {
            font-size: 1.1rem;
            font-weight: 600;
            color: var(--text-primary);
            margin-bottom: 0.25rem;
        }
        
        .customer-meta {
            font-size: 0.85rem;
            color: var(--text-secondary);
            margin-bottom: 0.5rem;
        }
        
        .enquiry-message {
            color: var(--text-primary);
            line-height: 1.6;
            margin: 1rem 0;
            padding: 1rem;
            background: var(--bg-light);
            border-radius: var(--radius-md);
        }
        
        .enquiry-footer {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-top: 1.5rem;
        }
        
        .status-badge {
            padding: 0.25rem 0.75rem;
            border-radius: 20px;
            font-size: 0.7rem;
            font-weight: 500;
            text-transform: uppercase;
        }
        
        .status-new {
            background: rgba(59, 130, 246, 0.1);
            color: var(--info-blue);
        }
        
        .status-in_progress {
            background: rgba(245, 158, 11, 0.1);
            color: var(--warning-yellow);
        }
        
        .status-resolved {
            background: rgba(16, 185, 129, 0.1);
            color: var(--success-green);
        }
        
        .status-closed {
            background: rgba(107, 114, 128, 0.1);
            color: var(--text-secondary);
        }
        
        .enquiry-actions {
            display: flex;
            gap: 0.5rem;
            flex-wrap: wrap;
        }
        
        .enquiry-detail {
            background: var(--bg-white);
            border-radius: var(--radius-lg);
            box-shadow: var(--shadow-md);
            overflow: hidden;
        }
        
        .enquiry-detail-header {
            background: var(--primary-purple);
            color: white;
            padding: 2rem;
            text-align: center;
        }
        
        .enquiry-detail-content {
            padding: 2rem;
        }
        
        .detail-section {
            margin-bottom: 2rem;
            padding-bottom: 1.5rem;
            border-bottom: 1px solid var(--border-color);
        }
        
        .detail-section:last-child {
            border-bottom: none;
            margin-bottom: 0;
        }
        
        .detail-title {
            font-size: 1.1rem;
            font-weight: 600;
            color: var(--text-primary);
            margin-bottom: 1rem;
        }
        
        .form-group {
            margin-bottom: 1.5rem;
        }
        
        .form-label {
            display: block;
            font-weight: 500;
            color: var(--text-primary);
            margin-bottom: 0.5rem;
            font-size: 0.9rem;
        }
        
        .form-input {
            width: 100%;
            padding: 0.75rem;
            border: 1px solid var(--border-color);
            border-radius: var(--radius-md);
            font-size: 1rem;
            transition: var(--transition);
        }
        
        .form-input:focus {
            outline: none;
            border-color: var(--primary-purple);
            box-shadow: 0 0 0 3px rgba(107, 70, 193, 0.1);
        }
        
        .form-textarea { min-height: 120px; resize: vertical; }
        
        .form-checkbox {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            margin: 1rem 0;
        }
        
        .form-checkbox input[type="checkbox"] {
            width: 18px;
            height: 18px;
        }
        
        .empty-state {
            padding: 3rem;
            text-align: center;
            color: var(--text-secondary);
        }
        
        .empty-state i {
            font-size: 3rem;
            margin-bottom: 1rem;
            opacity: 0.5;
        }
        
        .admin-notes {
            background: var(--bg-light);
            padding: 1rem;
            border-radius: var(--radius-md);
            margin: 1rem 0;
            font-size: 0.9rem;
        }
        
        .admin-response {
            background: rgba(16, 185, 129, 0.05);
            border: 1px solid rgba(16, 185, 129, 0.2);
            padding: 1rem;
            border-radius: var(--radius-md);
            margin: 1rem 0;
        }
        
        @media (max-width: 768px) {
            .sidebar { width: 100%; height: auto; position: relative; }
            .main-content { margin-left: 0; padding: 1rem; }
            .page-header { flex-direction: column; gap: 1rem; align-items: flex-start; }
            .filters-section { flex-direction: column; align-items: stretch; }
            .search-input { width: 100%; }
            .filter-tabs { flex-wrap: wrap; }
        }
    </style>
</head>
<body>
    <div class="admin-layout">
        <!-- Sidebar -->
        <div class="sidebar">
            <div class="sidebar-header">
                <div class="sidebar-logo">
                    <div class="logo-icon">
                        <i class="fas fa-crown"></i>
                    </div>
                    <div>
                        <div class="sidebar-title">SSV Admin</div>
                        <div class="sidebar-subtitle">Event Management</div>
                    </div>
                </div>
            </div>
            
            <nav class="sidebar-nav">
                <div class="nav-item">
                    <a href="dashboard.php" class="nav-link">
                        <i class="fas fa-tachometer-alt"></i>
                        Dashboard
                    </a>
                </div>
                <div class="nav-item">
                    <a href="hero-slides.php" class="nav-link">
                        <i class="fas fa-images"></i>
                        Hero Slider
                    </a>
                </div>
                <div class="nav-item">
                    <a href="services.php" class="nav-link">
                        <i class="fas fa-concierge-bell"></i>
                        Services
                    </a>
                </div>
                <div class="nav-item">
                    <a href="gallery.php" class="nav-link">
                        <i class="fas fa-photo-video"></i>
                        Gallery
                    </a>
                </div>
                <div class="nav-item">
                    <a href="testimonials.php" class="nav-link">
                        <i class="fas fa-star"></i>
                        Testimonials
                    </a>
                </div>
                <div class="nav-item">
                    <a href="enquiries.php" class="nav-link active">
                        <i class="fas fa-envelope"></i>
                        Enquiries
                    </a>
                </div>
                <div class="nav-item">
                    <a href="settings.php" class="nav-link">
                        <i class="fas fa-cog"></i>
                        Settings
                    </a>
                </div>
            </nav>
        </div>
        
        <!-- Main Content -->
        <div class="main-content">
            <div class="page-header">
                <h1 class="page-title"><?php echo $page_title; ?></h1>
                <div class="admin-info">
                    <div class="admin-avatar">
                        <?php echo strtoupper(substr($admin['name'], 0, 1)); ?>
                    </div>
                    <div class="admin-details">
                        <div class="admin-name"><?php echo htmlspecialchars($admin['name']); ?></div>
                        <div class="admin-role">Administrator</div>
                    </div>
                    <form method="POST" style="margin-left: 1rem;">
                        <button type="submit" name="logout" class="logout-btn">
                            <i class="fas fa-sign-out-alt"></i> Logout
                        </button>
                    </form>
                </div>
            </div>
            
            <?php if ($message): ?>
                <div class="alert alert-<?php echo $message_type; ?>">
                    <i class="fas fa-<?php echo $message_type === 'success' ? 'check-circle' : 'exclamation-triangle'; ?>"></i>
                    <?php echo htmlspecialchars($message); ?>
                </div>
            <?php endif; ?>
            
            <!-- Enquiry Detail View -->
            <?php if ($view_enquiry): ?>
            <div class="enquiry-detail">
                <div class="enquiry-detail-header">
                    <h2><i class="fas fa-envelope"></i> Enquiry Details</h2>
                    <p>Enquiry #<?php echo $view_enquiry['id']; ?></p>
                </div>
                
                <div class="enquiry-detail-content">
                    <div class="detail-section">
                        <h3 class="detail-title">Customer Information</h3>
                        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 1rem;">
                            <div><strong>Name:</strong> <?php echo htmlspecialchars($view_enquiry['name']); ?></div>
                            <div><strong>Email:</strong> <?php echo htmlspecialchars($view_enquiry['email']); ?></div>
                            <div><strong>Phone:</strong> <?php echo htmlspecialchars($view_enquiry['phone'] ?: 'Not provided'); ?></div>
                            <div><strong>Event Type:</strong> <?php echo htmlspecialchars($view_enquiry['event_type'] ?: 'Not specified'); ?></div>
                            <?php if ($view_enquiry['event_date']): ?>
                                <div><strong>Event Date:</strong> <?php echo date('M d, Y', strtotime($view_enquiry['event_date'])); ?></div>
                            <?php endif; ?>
                            <div><strong>Status:</strong> 
                                <span class="status-badge status-<?php echo $view_enquiry['status']; ?>">
                                    <?php echo $status_labels[$view_enquiry['status']] ?? ucfirst($view_enquiry['status']); ?>
                                </span>
                            </div>
                        </div>
                    </div>
                    
                    <div class="detail-section">
                        <h3 class="detail-title">Enquiry Message</h3>
                        <div class="enquiry-message">
                            <?php echo nl2br(htmlspecialchars($view_enquiry['message'])); ?>
                        </div>
                        <small class="text-muted">
                            Submitted: <?php echo date('M d, Y g:i A', strtotime($view_enquiry['created_at'])); ?>
                        </small>
                    </div>
                    
                    <!-- Admin Notes -->
                    <?php if (!empty($view_enquiry['admin_notes'])): ?>
                    <div class="detail-section">
                        <h3 class="detail-title">Admin Notes</h3>
                        <div class="admin-notes">
                            <?php echo nl2br(htmlspecialchars($view_enquiry['admin_notes'])); ?>
                        </div>
                    </div>
                    <?php endif; ?>
                    
                    <!-- Admin Response -->
                    <?php if (!empty($view_enquiry['admin_response'])): ?>
                    <div class="detail-section">
                        <h3 class="detail-title">Admin Response</h3>
                        <div class="admin-response">
                            <strong>Response sent to customer:</strong><br>
                            <?php echo nl2br(htmlspecialchars($view_enquiry['admin_response'])); ?>
                        </div>
                    </div>
                    <?php endif; ?>
                    
                    <!-- Status Update -->
                    <div class="detail-section">
                        <h3 class="detail-title">Update Status</h3>
                        <form method="POST" style="display: inline-flex; gap: 0.5rem; flex-wrap: wrap;">
                            <input type="hidden" name="enquiry_id" value="<?php echo $view_enquiry['id']; ?>">
                            <?php foreach ($statuses as $status): ?>
                                <?php if ($status !== $view_enquiry['status']): ?>
                                    <button type="submit" name="update_status" value="<?php echo $status; ?>" 
                                            class="btn btn-sm btn-<?php echo $status === 'resolved' ? 'success' : ($status === 'closed' ? 'secondary' : 'warning'); ?>">
                                        <i class="fas fa-<?php echo $status === 'resolved' ? 'check' : ($status === 'closed' ? 'times' : 'clock'); ?>"></i>
                                        Mark as <?php echo $status_labels[$status]; ?>
                                    </button>
                                    <input type="hidden" name="new_status" value="<?php echo $status; ?>">
                                <?php endif; ?>
                            <?php endforeach; ?>
                        </form>
                    </div>
                    
                    <!-- Add Note -->
                    <div class="detail-section">
                        <h3 class="detail-title">Add Admin Note</h3>
                        <form method="POST">
                            <input type="hidden" name="enquiry_id" value="<?php echo $view_enquiry['id']; ?>">
                            <div class="form-group">
                                <textarea name="admin_note" class="form-input form-textarea" 
                                          placeholder="Add internal notes about this enquiry..." required></textarea>
                            </div>
                            <button type="submit" name="add_note" class="btn btn-secondary">
                                <i class="fas fa-sticky-note"></i> Add Note
                            </button>
                        </form>
                    </div>
                    
                    <!-- Send Response -->
                    <div class="detail-section">
                        <h3 class="detail-title">Send Response to Customer</h3>
                        <form method="POST">
                            <input type="hidden" name="enquiry_id" value="<?php echo $view_enquiry['id']; ?>">
                            <div class="form-group">
                                <label for="response_message" class="form-label">Response Message *</label>
                                <textarea id="response_message" name="response_message" class="form-input form-textarea" 
                                          placeholder="Type your response to the customer..." required><?php echo htmlspecialchars($view_enquiry['admin_response'] ?? ''); ?></textarea>
                            </div>
                            <div class="form-checkbox">
                                <input type="checkbox" id="mark_resolved" name="mark_resolved">
                                <label for="mark_resolved" class="form-label" style="margin-bottom: 0;">Mark as resolved after sending</label>
                            </div>
                            <div style="display: flex; gap: 1rem;">
                                <button type="submit" name="send_response" class="btn btn-primary">
                                    <i class="fas fa-paper-plane"></i> Send Response
                                </button>
                                <a href="enquiries.php" class="btn btn-secondary">
                                    <i class="fas fa-arrow-left"></i> Back to List
                                </a>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
            <?php else: ?>
            
            <!-- Filters and Search -->
            <div class="filters-section">
                <div class="filter-tabs">
                    <a href="enquiries.php?filter=all<?php echo !empty($_GET['search']) ? '&search=' . urlencode($_GET['search']) : ''; ?>" 
                       class="filter-tab <?php echo ($filter === 'all') ? 'active' : ''; ?>">
                        All (<?php echo $status_counts['all'] ?? 0; ?>)
                    </a>
                    <?php foreach ($statuses as $status): ?>
                        <a href="enquiries.php?filter=<?php echo $status; ?><?php echo !empty($_GET['search']) ? '&search=' . urlencode($_GET['search']) : ''; ?>" 
                           class="filter-tab <?php echo ($filter === $status) ? 'active' : ''; ?>">
                            <?php echo $status_labels[$status]; ?> (<?php echo $status_counts[$status] ?? 0; ?>)
                        </a>
                    <?php endforeach; ?>
                </div>
                
                <form method="GET" class="search-form">
                    <input type="hidden" name="filter" value="<?php echo htmlspecialchars($filter); ?>">
                    <input type="text" name="search" class="search-input" placeholder="Search enquiries..." 
                           value="<?php echo htmlspecialchars($search); ?>">
                    <button type="submit" class="btn btn-secondary">
                        <i class="fas fa-search"></i> Search
                    </button>
                </form>
            </div>
            
            <!-- Enquiries List -->
            <div class="form-section">
                <h2 class="section-title">Customer Enquiries</h2>
                
                <?php if (empty($enquiries)): ?>
                    <div class="empty-state">
                        <i class="fas fa-envelope"></i>
                        <h3>No enquiries found</h3>
                        <p>No enquiries match your current filters.</p>
                    </div>
                <?php else: ?>
                    <div class="enquiry-grid">
                        <?php foreach ($enquiries as $enquiry): ?>
                            <div class="enquiry-card">
                                <div class="enquiry-header">
                                    <div class="customer-info">
                                        <h3><?php echo htmlspecialchars($enquiry['name']); ?></h3>
                                        <div class="customer-meta">
                                            <span><i class="fas fa-envelope"></i> <?php echo htmlspecialchars($enquiry['email']); ?></span>
                                            <?php if ($enquiry['phone']): ?>
                                                <span><i class="fas fa-phone"></i> <?php echo htmlspecialchars($enquiry['phone']); ?></span>
                                            <?php endif; ?>
                                            <?php if ($enquiry['event_type']): ?>
                                                <span><i class="fas fa-calendar"></i> <?php echo htmlspecialchars($enquiry['event_type']); ?></span>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                    <div>
                                        <span class="status-badge status-<?php echo $enquiry['status']; ?>">
                                            <?php echo $status_labels[$enquiry['status']] ?? ucfirst($enquiry['status']); ?>
                                        </span>
                                    </div>
                                </div>
                                
                                <div class="enquiry-message">
                                    <?php echo nl2br(htmlspecialchars(substr($enquiry['message'], 0, 200))); ?>
                                    <?php if (strlen($enquiry['message']) > 200): ?>...<?php endif; ?>
                                </div>
                                
                                <?php if (!empty($enquiry['admin_response'])): ?>
                                    <div class="admin-response">
                                        <strong>Response sent:</strong> 
                                        <?php echo htmlspecialchars(substr($enquiry['admin_response'], 0, 100)); ?>...
                                    </div>
                                <?php endif; ?>
                                
                                <div class="enquiry-footer">
                                    <small class="text-muted">
                                        Received: <?php echo date('M d, Y g:i A', strtotime($enquiry['created_at'])); ?>
                                    </small>
                                    
                                    <div class="enquiry-actions">
                                        <a href="enquiries.php?view=<?php echo $enquiry['id']; ?>" class="btn btn-info btn-sm">
                                            <i class="fas fa-eye"></i> View Details
                                        </a>
                                        
                                        <?php if ($enquiry['status'] === 'new'): ?>
                                            <form method="POST" style="display: inline;">
                                                <input type="hidden" name="enquiry_id" value="<?php echo $enquiry['id']; ?>">
                                                <input type="hidden" name="new_status" value="in_progress">
                                                <button type="submit" name="update_status" class="btn btn-warning btn-sm">
                                                    <i class="fas fa-clock"></i> Start Processing
                                                </button>
                                            </form>
                                        <?php endif; ?>
                                        
                                        <form method="POST" style="display: inline;" onsubmit="return confirm('Delete this enquiry?');">
                                            <input type="hidden" name="enquiry_id" value="<?php echo $enquiry['id']; ?>">
                                            <button type="submit" name="delete_enquiry" class="btn btn-danger btn-sm">
                                                <i class="fas fa-trash"></i> Delete
                                            </button>
                                        </form>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
            </div>
            
            <?php endif; ?>
        </div>
    </div>
</body>
</html>