<?php
session_start();

// Include required files
require_once '../config/db.php';
require_once 'auth.php';

// Check if admin is logged in
requireAdminLogin();

// Get admin info
$admin = getCurrentAdmin();

// Handle logout
if (isset($_POST['logout'])) {
    adminLogout();
}

// Get dashboard statistics
$stats = [];
try {
    $db = getDB();
    
    // Get total services
    $stmt = $db->query("SELECT COUNT(*) as count FROM services WHERE is_active = 1");
    $stats['services'] = $stmt->fetch()['count'];
    
    // Get total gallery items
    $stmt = $db->query("SELECT COUNT(*) as count FROM gallery");
    $stats['gallery'] = $stmt->fetch()['count'];
    
    // Get total testimonials
    $stmt = $db->query("SELECT COUNT(*) as count FROM testimonials WHERE is_approved = 1");
    $stats['testimonials'] = $stmt->fetch()['count'];
    
    // Get total enquiries
    $stmt = $db->query("SELECT COUNT(*) as count FROM enquiries");
    $stats['enquiries'] = $stmt->fetch()['count'];
    
    // Get new enquiries
    $stmt = $db->query("SELECT COUNT(*) as count FROM enquiries WHERE status = 'new'");
    $stats['new_enquiries'] = $stmt->fetch()['count'];
    
    // Get hero slides
    $stmt = $db->query("SELECT COUNT(*) as count FROM hero_slides WHERE is_active = 1");
    $stats['hero_slides'] = $stmt->fetch()['count'];
    
    // Get recent enquiries
    $stmt = $db->query("SELECT * FROM enquiries ORDER BY created_at DESC LIMIT 5");
    $recent_enquiries = $stmt->fetchAll();
    
} catch (Exception $e) {
    $stats = ['services' => 0, 'gallery' => 0, 'testimonials' => 0, 'enquiries' => 0, 'new_enquiries' => 0, 'hero_slides' => 0];
    $recent_enquiries = [];
}

$page_title = 'Dashboard';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $page_title; ?> - Admin Panel</title>
    <link rel="icon" type="image/x-icon" href="../public/favicon.ico">
    
    <!-- Google Fonts -->
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    
    <!-- Font Awesome -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    
    <style>
        :root {
            --primary-purple: #6B46C1;
            --primary-gold: #B8860B;
            --primary-maroon: #8B1538;
            --text-primary: #1F2937;
            --text-secondary: #6B7280;
            --bg-light: #F9FAFB;
            --bg-white: #FFFFFF;
            --border-color: #E5E7EB;
            --success-green: #10B981;
            --warning-yellow: #F59E0B;
            --error-red: #EF4444;
            --shadow-sm: 0 1px 2px 0 rgba(0, 0, 0, 0.05);
            --shadow-md: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
            --shadow-lg: 0 10px 15px -3px rgba(0, 0, 0, 0.1), 0 4px 6px -2px rgba(0, 0, 0, 0.05);
            --radius-md: 8px;
            --radius-lg: 12px;
            --transition: all 0.3s ease;
        }
        
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Poppins', sans-serif;
            background: var(--bg-light);
            color: var(--text-primary);
            line-height: 1.6;
        }
        
        .admin-layout {
            display: flex;
            min-height: 100vh;
        }
        
        /* Sidebar */
        .sidebar {
            width: 280px;
            background: var(--bg-white);
            border-right: 1px solid var(--border-color);
            padding: 2rem 0;
            position: fixed;
            height: 100vh;
            overflow-y: auto;
            box-shadow: var(--shadow-sm);
        }
        
        .sidebar-header {
            padding: 0 2rem 2rem;
            border-bottom: 1px solid var(--border-color);
            margin-bottom: 2rem;
        }
        
        .sidebar-logo {
            display: flex;
            align-items: center;
            gap: 1rem;
        }
        
        .sidebar-logo .logo-icon {
            width: 40px;
            height: 40px;
            background: var(--primary-gold);
            border-radius: 8px;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
        }
        
        .sidebar-title {
            font-size: 1.2rem;
            font-weight: 600;
            color: var(--text-primary);
        }
        
        .sidebar-subtitle {
            font-size: 0.8rem;
            color: var(--text-secondary);
        }
        
        .sidebar-nav {
            padding: 0 1rem;
        }
        
        .nav-item {
            margin-bottom: 0.5rem;
        }
        
        .nav-link {
            display: flex;
            align-items: center;
            gap: 1rem;
            padding: 1rem;
            color: var(--text-secondary);
            text-decoration: none;
            border-radius: var(--radius-md);
            transition: var(--transition);
            font-weight: 500;
        }
        
        .nav-link:hover {
            background: var(--bg-light);
            color: var(--primary-purple);
        }
        
        .nav-link.active {
            background: var(--primary-purple);
            color: white;
        }
        
        .nav-link i {
            width: 20px;
            text-align: center;
        }
        
        /* Main Content */
        .main-content {
            flex: 1;
            margin-left: 280px;
            padding: 2rem;
        }
        
        .page-header {
            display: flex;
            justify-content: between;
            align-items: center;
            margin-bottom: 2rem;
            padding-bottom: 1rem;
            border-bottom: 1px solid var(--border-color);
        }
        
        .page-title {
            font-size: 2rem;
            font-weight: 600;
            color: var(--text-primary);
        }
        
        .admin-info {
            display: flex;
            align-items: center;
            gap: 1rem;
        }
        
        .admin-avatar {
            width: 40px;
            height: 40px;
            background: var(--primary-gold);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-weight: 600;
        }
        
        .admin-details {
            display: flex;
            flex-direction: column;
        }
        
        .admin-name {
            font-weight: 600;
            font-size: 0.9rem;
        }
        
        .admin-role {
            font-size: 0.8rem;
            color: var(--text-secondary);
        }
        
        .logout-btn {
            background: var(--error-red);
            color: white;
            border: none;
            padding: 0.5rem 1rem;
            border-radius: var(--radius-md);
            cursor: pointer;
            font-size: 0.85rem;
            transition: var(--transition);
        }
        
        .logout-btn:hover {
            background: #DC2626;
        }
        
        /* Stats Cards */
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 1.5rem;
            margin-bottom: 3rem;
        }
        
        .stat-card {
            background: var(--bg-white);
            padding: 2rem;
            border-radius: var(--radius-lg);
            box-shadow: var(--shadow-md);
            transition: var(--transition);
            border-left: 4px solid transparent;
        }
        
        .stat-card:hover {
            transform: translateY(-2px);
            box-shadow: var(--shadow-lg);
        }
        
        .stat-card.purple { border-left-color: var(--primary-purple); }
        .stat-card.gold { border-left-color: var(--primary-gold); }
        .stat-card.green { border-left-color: var(--success-green); }
        .stat-card.red { border-left-color: var(--error-red); }
        
        .stat-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 1rem;
        }
        
        .stat-title {
            font-size: 0.9rem;
            font-weight: 500;
            color: var(--text-secondary);
        }
        
        .stat-icon {
            width: 40px;
            height: 40px;
            border-radius: 8px;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
        }
        
        .stat-icon.purple { background: var(--primary-purple); }
        .stat-icon.gold { background: var(--primary-gold); }
        .stat-icon.green { background: var(--success-green); }
        .stat-icon.red { background: var(--error-red); }
        
        .stat-value {
            font-size: 2.5rem;
            font-weight: 700;
            color: var(--text-primary);
            margin-bottom: 0.5rem;
        }
        
        .stat-description {
            font-size: 0.85rem;
            color: var(--text-secondary);
        }
        
        /* Recent Activity */
        .recent-section {
            background: var(--bg-white);
            border-radius: var(--radius-lg);
            box-shadow: var(--shadow-md);
            overflow: hidden;
        }
        
        .section-header {
            padding: 1.5rem 2rem;
            border-bottom: 1px solid var(--border-color);
            background: var(--bg-light);
        }
        
        .section-title {
            font-size: 1.2rem;
            font-weight: 600;
            color: var(--text-primary);
        }
        
        .enquiry-list {
            padding: 0;
        }
        
        .enquiry-item {
            padding: 1.5rem 2rem;
            border-bottom: 1px solid var(--border-color);
            transition: var(--transition);
        }
        
        .enquiry-item:hover {
            background: var(--bg-light);
        }
        
        .enquiry-item:last-child {
            border-bottom: none;
        }
        
        .enquiry-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 0.5rem;
        }
        
        .enquiry-name {
            font-weight: 600;
            color: var(--text-primary);
        }
        
        .enquiry-date {
            font-size: 0.8rem;
            color: var(--text-secondary);
        }
        
        .enquiry-details {
            display: flex;
            gap: 2rem;
            font-size: 0.85rem;
            color: var(--text-secondary);
        }
        
        .status-badge {
            padding: 0.25rem 0.75rem;
            border-radius: 20px;
            font-size: 0.75rem;
            font-weight: 500;
            text-transform: uppercase;
        }
        
        .status-new {
            background: rgba(239, 68, 68, 0.1);
            color: var(--error-red);
        }
        
        .status-read {
            background: rgba(245, 158, 11, 0.1);
            color: var(--warning-yellow);
        }
        
        .empty-state {
            padding: 3rem;
            text-align: center;
            color: var(--text-secondary);
        }
        
        .empty-state i {
            font-size: 3rem;
            margin-bottom: 1rem;
            color: var(--text-secondary);
            opacity: 0.5;
        }
        
        @media (max-width: 768px) {
            .sidebar {
                width: 100%;
                height: auto;
                position: relative;
            }
            
            .main-content {
                margin-left: 0;
                padding: 1rem;
            }
            
            .stats-grid {
                grid-template-columns: 1fr;
            }
            
            .page-header {
                flex-direction: column;
                gap: 1rem;
                align-items: flex-start;
            }
            
            .admin-info {
                width: 100%;
                justify-content: space-between;
            }
        }
    </style>
</head>
<body>
    <div class="admin-layout">
        <!-- Sidebar -->
        <div class="sidebar">
            <div class="sidebar-header">
                <div class="sidebar-logo">
                    <div class="logo-icon">
                        <i class="fas fa-crown"></i>
                    </div>
                    <div>
                        <div class="sidebar-title">SSV Admin</div>
                        <div class="sidebar-subtitle">Event Management</div>
                    </div>
                </div>
            </div>
            
            <nav class="sidebar-nav">
                <div class="nav-item">
                    <a href="dashboard.php" class="nav-link active">
                        <i class="fas fa-tachometer-alt"></i>
                        Dashboard
                    </a>
                </div>
                <div class="nav-item">
                    <a href="hero-slides.php" class="nav-link">
                        <i class="fas fa-images"></i>
                        Hero Slider
                    </a>
                </div>
                <div class="nav-item">
                    <a href="services.php" class="nav-link">
                        <i class="fas fa-concierge-bell"></i>
                        Services
                    </a>
                </div>
                <div class="nav-item">
                    <a href="gallery.php" class="nav-link">
                        <i class="fas fa-photo-video"></i>
                        Gallery
                    </a>
                </div>
                <div class="nav-item">
                    <a href="testimonials.php" class="nav-link">
                        <i class="fas fa-star"></i>
                        Testimonials
                    </a>
                </div>
                <div class="nav-item">
                    <a href="enquiries.php" class="nav-link">
                        <i class="fas fa-envelope"></i>
                        Enquiries
                        <?php if ($stats['new_enquiries'] > 0): ?>
                            <span class="status-badge status-new"><?php echo $stats['new_enquiries']; ?></span>
                        <?php endif; ?>
                    </a>
                </div>
                <div class="nav-item">
                    <a href="settings.php" class="nav-link">
                        <i class="fas fa-cog"></i>
                        Settings
                    </a>
                </div>
            </nav>
        </div>
        
        <!-- Main Content -->
        <div class="main-content">
            <div class="page-header">
                <h1 class="page-title"><?php echo $page_title; ?></h1>
                <div class="admin-info">
                    <div class="admin-avatar">
                        <?php echo strtoupper(substr($admin['name'], 0, 1)); ?>
                    </div>
                    <div class="admin-details">
                        <div class="admin-name"><?php echo htmlspecialchars($admin['name']); ?></div>
                        <div class="admin-role">Administrator</div>
                    </div>
                    <form method="POST" style="margin-left: 1rem;">
                        <button type="submit" name="logout" class="logout-btn">
                            <i class="fas fa-sign-out-alt"></i> Logout
                        </button>
                    </form>
                </div>
            </div>
            
            <!-- Stats Cards -->
            <div class="stats-grid">
                <div class="stat-card purple">
                    <div class="stat-header">
                        <div class="stat-title">Hero Slides</div>
                        <div class="stat-icon purple">
                            <i class="fas fa-images"></i>
                        </div>
                    </div>
                    <div class="stat-value"><?php echo $stats['hero_slides']; ?></div>
                    <div class="stat-description">Active slider images</div>
                </div>
                
                <div class="stat-card gold">
                    <div class="stat-header">
                        <div class="stat-title">Services</div>
                        <div class="stat-icon gold">
                            <i class="fas fa-concierge-bell"></i>
                        </div>
                    </div>
                    <div class="stat-value"><?php echo $stats['services']; ?></div>
                    <div class="stat-description">Active services</div>
                </div>
                
                <div class="stat-card green">
                    <div class="stat-header">
                        <div class="stat-title">Gallery</div>
                        <div class="stat-icon green">
                            <i class="fas fa-photo-video"></i>
                        </div>
                    </div>
                    <div class="stat-value"><?php echo $stats['gallery']; ?></div>
                    <div class="stat-description">Total media items</div>
                </div>
                
                <div class="stat-card red">
                    <div class="stat-header">
                        <div class="stat-title">New Enquiries</div>
                        <div class="stat-icon red">
                            <i class="fas fa-envelope"></i>
                        </div>
                    </div>
                    <div class="stat-value"><?php echo $stats['new_enquiries']; ?></div>
                    <div class="stat-description">Pending responses</div>
                </div>
            </div>
            
            <!-- Recent Enquiries -->
            <div class="recent-section">
                <div class="section-header">
                    <h2 class="section-title">Recent Enquiries</h2>
                </div>
                
                <?php if (empty($recent_enquiries)): ?>
                    <div class="empty-state">
                        <i class="fas fa-inbox"></i>
                        <h3>No enquiries yet</h3>
                        <p>New customer enquiries will appear here.</p>
                    </div>
                <?php else: ?>
                    <div class="enquiry-list">
                        <?php foreach ($recent_enquiries as $enquiry): ?>
                            <div class="enquiry-item">
                                <div class="enquiry-header">
                                    <div class="enquiry-name"><?php echo htmlspecialchars($enquiry['name']); ?></div>
                                    <div class="enquiry-date"><?php echo formatAdminDate($enquiry['created_at']); ?></div>
                                </div>
                                <div class="enquiry-details">
                                    <span><i class="fas fa-envelope"></i> <?php echo htmlspecialchars($enquiry['email']); ?></span>
                                    <span><i class="fas fa-phone"></i> <?php echo htmlspecialchars($enquiry['phone']); ?></span>
                                    <?php if ($enquiry['service_interested']): ?>
                                        <span><i class="fas fa-concierge-bell"></i> <?php echo htmlspecialchars($enquiry['service_interested']); ?></span>
                                    <?php endif; ?>
                                    <span class="status-badge status-<?php echo $enquiry['status']; ?>">
                                        <?php echo ucfirst($enquiry['status']); ?>
                                    </span>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</body>
</html>